<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Midia;
use App\Models\Empresa;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class MidiaPerformanceTest extends TestCase
{
    use RefreshDatabase;

    protected $user;
    protected $empresa;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
        $this->empresa = Empresa::factory()->create();
        $this->actingAs($this->user);
        Storage::fake('public');
    }

    public function test_bulk_create_performance()
    {
        $startTime = microtime(true);
        
        // Criar 50 mídias em lote
        for ($i = 1; $i <= 50; $i++) {
            $response = $this->post(route('midias.store'), [
                'title' => "Mídia Performance {$i}",
                'type' => 'image',
                'empresa_id' => $this->empresa->id,
                'equipment_serial' => "PERF-{$i}",
                'position' => ($i % 6) + 1,
            ]);
            
            $response->assertRedirect(route('midias.index'));
        }
        
        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;
        
        // Verificar se todas foram criadas
        $this->assertEquals(50, Midia::count());
        
        // Performance deve ser razoável (menos de 30 segundos para 50 items)
        $this->assertLessThan(30, $executionTime, 
            "Criação de 50 mídias levou {$executionTime} segundos, esperado < 30s");
    }

    public function test_large_file_upload_handling()
    {
        // Simular arquivo grande (10MB)
        $file = UploadedFile::fake()->image('large-file.jpg', 4000, 3000)->size(10240);

        $startTime = microtime(true);
        
        $response = $this->post(route('midias.store'), [
            'title' => 'Large File Test',
            'type' => 'image',
            'media_file' => $file,
            'empresa_id' => $this->empresa->id,
        ]);
        
        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;

        $response->assertRedirect(route('midias.index'));
        
        // Verificar se o arquivo foi processado
        $midia = Midia::where('title', 'Large File Test')->first();
        $this->assertNotNull($midia);
        $this->assertNotNull($midia->path);
        
        // Verificar se foi redimensionado para 640x1100
        $this->assertTrue(Storage::disk('public')->exists($midia->path));
        
        // Upload e processamento de arquivo grande deve ser razoável (< 10s)
        $this->assertLessThan(10, $executionTime,
            "Upload de arquivo grande levou {$executionTime} segundos, esperado < 10s");
    }

    public function test_pagination_performance_with_large_dataset()
    {
        // Criar muitas mídias (100)
        Midia::factory()->count(100)->create([
            'empresa_id' => $this->empresa->id,
        ]);

        $startTime = microtime(true);
        
        // Testar primeira página
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        
        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;
        
        // Paginação deve ser rápida mesmo com muitos registros (< 2s)
        $this->assertLessThan(2, $executionTime,
            "Paginação com 100 registros levou {$executionTime} segundos, esperado < 2s");
        
        // Testar busca com dataset grande
        $startTime = microtime(true);
        
        $response = $this->get(route('midias.index', ['q' => 'Test']));
        $response->assertStatus(200);
        
        $endTime = microtime(true);
        $searchTime = $endTime - $startTime;
        
        // Busca deve ser rápida (< 3s)
        $this->assertLessThan(3, $searchTime,
            "Busca em 100 registros levou {$searchTime} segundos, esperado < 3s");
    }

    public function test_concurrent_uploads_simulation()
    {
        $files = [];
        $responses = [];
        
        // Preparar múltiplos uploads
        for ($i = 1; $i <= 5; $i++) {
            $files[$i] = UploadedFile::fake()->image("concurrent-{$i}.jpg", 800, 600);
        }
        
        $startTime = microtime(true);
        
        // Simular uploads concorrentes
        foreach ($files as $i => $file) {
            $responses[$i] = $this->post(route('midias.store'), [
                'title' => "Concurrent Upload {$i}",
                'type' => 'image',
                'media_file' => $file,
                'empresa_id' => $this->empresa->id,
                'position' => $i,
            ]);
        }
        
        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;
        
        // Verificar se todos uploads foram bem-sucedidos
        foreach ($responses as $response) {
            $response->assertRedirect(route('midias.index'));
        }
        
        $this->assertEquals(5, Midia::count());
        
        // Múltiplos uploads devem ser processados em tempo razoável (< 15s)
        $this->assertLessThan(15, $executionTime,
            "5 uploads concorrentes levaram {$executionTime} segundos, esperado < 15s");
    }

    public function test_memory_usage_during_image_processing()
    {
        $initialMemory = memory_get_usage();
        
        // Processar múltiplas imagens
        for ($i = 1; $i <= 10; $i++) {
            $file = UploadedFile::fake()->image("memory-test-{$i}.jpg", 2000, 1500);
            
            $response = $this->post(route('midias.store'), [
                'title' => "Memory Test {$i}",
                'type' => 'image',
                'media_file' => $file,
                'empresa_id' => $this->empresa->id,
            ]);
            
            $response->assertRedirect(route('midias.index'));
        }
        
        $finalMemory = memory_get_usage();
        $memoryIncrease = $finalMemory - $initialMemory;
        
        // Verificar se todas as imagens foram processadas
        $this->assertEquals(10, Midia::count());
        
        // Aumento de memória deve ser razoável (< 50MB)
        $maxMemoryIncrease = 50 * 1024 * 1024; // 50MB
        $this->assertLessThan($maxMemoryIncrease, $memoryIncrease,
            "Aumento de memória foi " . number_format($memoryIncrease / 1024 / 1024, 2) . "MB, esperado < 50MB");
    }

    public function test_database_queries_optimization()
    {
        // Criar empresas e mídias relacionadas
        $empresas = Empresa::factory()->count(5)->create();
        
        foreach ($empresas as $empresa) {
            Midia::factory()->count(10)->create(['empresa_id' => $empresa->id]);
        }
        
        // Contar queries antes da consulta
        $queriesCount = 0;
        DB::listen(function ($query) use (&$queriesCount) {
            $queriesCount++;
        });
        
        // Acessar listagem com relacionamentos
        $response = $this->get(route('midias.index'));
        $response->assertStatus(200);
        
        // Número de queries deve ser otimizado (< 10 queries para 50 registros)
        $this->assertLessThan(10, $queriesCount,
            "Listagem executou {$queriesCount} queries, esperado < 10");
    }

    public function test_soft_delete_performance()
    {
        // Criar muitas mídias
        $midias = Midia::factory()->count(100)->create([
            'empresa_id' => $this->empresa->id,
        ]);
        
        $startTime = microtime(true);
        
        // Fazer soft delete de todas
        foreach ($midias as $midia) {
            $response = $this->delete(route('midias.destroy', $midia));
            $response->assertRedirect(route('midias.index'));
        }
        
        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;
        
        // Verificar se todos foram deletados
        $this->assertEquals(0, Midia::count()); // Contagem normal (não incluindo soft-deleted)
        
        // Soft delete em lote deve ser eficiente (< 20s para 100 registros)
        $this->assertLessThan(20, $executionTime,
            "Soft delete de 100 registros levou {$executionTime} segundos, esperado < 20s");
    }

    public function test_restore_performance()
    {
        // Criar e deletar mídias
        $midias = Midia::factory()->count(50)->create([
            'empresa_id' => $this->empresa->id,
        ]);
        
        foreach ($midias as $midia) {
            $midia->delete();
        }
        
        $startTime = microtime(true);
        
        // Restaurar todas
        foreach ($midias as $midia) {
            $response = $this->patch(route('midias.restore', $midia->id));
            $response->assertRedirect(route('midias.trash'));
        }
        
        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;
        
        // Verificar se todas foram restauradas
        $this->assertEquals(50, Midia::count());
        
        // Restauração deve ser eficiente (< 15s para 50 registros)
        $this->assertLessThan(15, $executionTime,
            "Restauração de 50 registros levou {$executionTime} segundos, esperado < 15s");
    }

    public function test_status_toggle_performance()
    {
        // Criar mídias
        $midias = Midia::factory()->count(100)->create([
            'empresa_id' => $this->empresa->id,
            'status' => 0, // Todas ativas inicialmente
        ]);
        
        $startTime = microtime(true);
        
        // Alternar status de todas
        foreach ($midias as $midia) {
            $response = $this->patch(route('midias.toggleStatus', $midia));
            $response->assertRedirect(route('midias.index'));
        }
        
        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;
        
        // Verificar se status foi alterado
        $inactiveMidias = Midia::where('status', 1)->count();
        $this->assertEquals(100, $inactiveMidias);
        
        // Toggle em lote deve ser rápido (< 10s para 100 registros)
        $this->assertLessThan(10, $executionTime,
            "Toggle de status de 100 registros levou {$executionTime} segundos, esperado < 10s");
    }
}