<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Midia;
use App\Models\Empresa;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

class MidiaUploadTest extends TestCase
{
    use RefreshDatabase;

    protected $user;
    protected $empresa;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
        $this->empresa = Empresa::factory()->create();
        $this->actingAs($this->user);
        Storage::fake('public');
    }

    public function test_can_upload_image_with_resize()
    {
        $file = UploadedFile::fake()->image('test-image.jpg', 1920, 1080);

        $response = $this->post(route('midias.store'), [
            'title' => 'Test Image Upload',
            'type' => 'image',
            'media_file' => $file,
            'empresa_id' => $this->empresa->id,
            'equipment_serial' => 'TEST-001',
            'position' => 1,
        ]);

        $response->assertRedirect(route('midias.index'));
        
        // Verificar se a mídia foi criada no banco
        $this->assertDatabaseHas('midias', [
            'title' => 'Test Image Upload',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
            'equipment_serial' => 'TEST-001',
            'position' => 1,
        ]);

        // Verificar se o arquivo foi processado e salvo
        $midia = Midia::where('title', 'Test Image Upload')->first();
        $this->assertNotNull($midia->path);
        $this->assertTrue(Storage::disk('public')->exists($midia->path));
    }

    public function test_can_upload_video_without_resize()
    {
        $file = UploadedFile::fake()->create('test-video.mp4', 1000, 'video/mp4');

        $response = $this->post(route('midias.store'), [
            'title' => 'Test Video Upload',
            'type' => 'video',
            'media_file' => $file,
            'empresa_id' => $this->empresa->id,
            'equipment_serial' => 'TEST-002',
            'position' => 2,
        ]);

        $response->assertRedirect(route('midias.index'));
        
        // Verificar se a mídia foi criada no banco
        $this->assertDatabaseHas('midias', [
            'title' => 'Test Video Upload',
            'type' => 'video',
            'empresa_id' => $this->empresa->id,
        ]);

        // Verificar se o arquivo foi salvo (vídeos não são redimensionados)
        $midia = Midia::where('title', 'Test Video Upload')->first();
        $this->assertNotNull($midia->path);
        $this->assertTrue(Storage::disk('public')->exists($midia->path));
    }

    public function test_can_create_midia_without_file()
    {
        $response = $this->post(route('midias.store'), [
            'title' => 'Test Media Without File',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
            'equipment_serial' => 'TEST-003',
            'position' => 3,
        ]);

        $response->assertRedirect(route('midias.index'));
        
        // Verificar se a mídia foi criada sem arquivo
        $this->assertDatabaseHas('midias', [
            'title' => 'Test Media Without File',
            'type' => 'image',
            'path' => null,
        ]);
    }

    public function test_can_update_midia_with_new_file()
    {
        // Criar mídia inicial com arquivo
        $initialFile = UploadedFile::fake()->image('initial.jpg');
        $midia = Midia::factory()->create([
            'title' => 'Initial Media',
            'type' => 'image',
            'path' => 'midias/initial.jpg',
            'empresa_id' => $this->empresa->id,
        ]);

        // Simular arquivo existente
        Storage::disk('public')->put($midia->path, 'fake content');
        
        // Fazer upload de novo arquivo
        $newFile = UploadedFile::fake()->image('updated.jpg', 800, 600);

        $response = $this->put(route('midias.update', $midia), [
            'title' => 'Updated Media',
            'type' => 'image',
            'media_file' => $newFile,
            'empresa_id' => $this->empresa->id,
            'equipment_serial' => 'TEST-004',
            'position' => 4,
        ]);

        $response->assertRedirect(route('midias.index'));
        
        // Verificar se a mídia foi atualizada
        $this->assertDatabaseHas('midias', [
            'id' => $midia->id,
            'title' => 'Updated Media',
        ]);

        // Verificar se o arquivo antigo foi removido e o novo foi salvo
        $updatedMidia = $midia->fresh();
        $this->assertNotEquals('midias/initial.jpg', $updatedMidia->path);
        $this->assertTrue(Storage::disk('public')->exists($updatedMidia->path));
        $this->assertFalse(Storage::disk('public')->exists('midias/initial.jpg'));
    }

    public function test_can_update_midia_without_changing_file()
    {
        $midia = Midia::factory()->create([
            'title' => 'Original Title',
            'type' => 'image',
            'path' => 'midias/original.jpg',
            'empresa_id' => $this->empresa->id,
        ]);

        // Simular arquivo existente
        Storage::disk('public')->put($midia->path, 'fake content');

        $response = $this->put(route('midias.update', $midia), [
            'title' => 'Updated Title Only',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
            'equipment_serial' => 'TEST-005',
            'position' => 5,
        ]);

        $response->assertRedirect(route('midias.index'));
        
        // Verificar se apenas o título foi alterado
        $this->assertDatabaseHas('midias', [
            'id' => $midia->id,
            'title' => 'Updated Title Only',
            'path' => 'midias/original.jpg', // Path deve permanecer o mesmo
        ]);

        // Verificar se o arquivo original ainda existe
        $this->assertTrue(Storage::disk('public')->exists('midias/original.jpg'));
    }

    public function test_image_is_resized_to_correct_dimensions()
    {
        $file = UploadedFile::fake()->image('large-image.jpg', 3000, 2000);

        $response = $this->post(route('midias.store'), [
            'title' => 'Large Image Test',
            'type' => 'image',
            'media_file' => $file,
            'empresa_id' => $this->empresa->id,
        ]);

        $response->assertRedirect(route('midias.index'));
        
        $midia = Midia::where('title', 'Large Image Test')->first();
        $this->assertNotNull($midia->path);
        
        // Verificar se o arquivo foi redimensionado
        $this->assertTrue(Storage::disk('public')->exists($midia->path));
        
        // Ler a imagem processada e verificar dimensões
        $imageContent = Storage::disk('public')->get($midia->path);
        $manager = new ImageManager(new Driver());
        $image = $manager->read($imageContent);
        
        $this->assertEquals(640, $image->width());
        $this->assertEquals(1100, $image->height());
    }

    public function test_validates_required_fields()
    {
        $response = $this->post(route('midias.store'), []);

        $response->assertSessionHasErrors(['title', 'type', 'empresa_id']);
    }

    public function test_validates_type_field()
    {
        $response = $this->post(route('midias.store'), [
            'title' => 'Test',
            'type' => 'invalid_type',
            'empresa_id' => $this->empresa->id,
        ]);

        $response->assertSessionHasErrors(['type']);
    }

    public function test_validates_position_range()
    {
        $response = $this->post(route('midias.store'), [
            'title' => 'Test',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
            'position' => 10, // Fora do range 1-6
        ]);

        $response->assertSessionHasErrors(['position']);
    }

    public function test_validates_empresa_exists()
    {
        $response = $this->post(route('midias.store'), [
            'title' => 'Test',
            'type' => 'image',
            'empresa_id' => 999999, // ID que não existe
        ]);

        $response->assertSessionHasErrors(['empresa_id']);
    }

    public function test_validates_datetime_range()
    {
        $response = $this->post(route('midias.store'), [
            'title' => 'Test',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
            'start_datetime' => '2025-12-31 23:59:59',
            'end_datetime' => '2025-01-01 00:00:00', // End before start
        ]);

        $response->assertSessionHasErrors(['end_datetime']);
    }

    public function test_userstamps_are_recorded_on_create()
    {
        $response = $this->post(route('midias.store'), [
            'title' => 'Userstamps Test',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
        ]);

        $midia = Midia::where('title', 'Userstamps Test')->first();
        
        $this->assertEquals($this->user->id, $midia->created_by);
        $this->assertEquals($this->user->id, $midia->updated_by);
        $this->assertNull($midia->deleted_by);
    }

    public function test_userstamps_are_recorded_on_update()
    {
        $otherUser = User::factory()->create();
        
        // Criar mídia com um usuário temporariamente fazendo login como esse usuário
        $this->actingAs($otherUser);
        $midia = Midia::factory()->create([
            'empresa_id' => $this->empresa->id,
        ]);
        
        // Verificar se foi criada pelo outro usuário
        $this->assertEquals($otherUser->id, $midia->created_by);
        $this->assertEquals($otherUser->id, $midia->updated_by);

        // Mudar para o usuário principal e atualizar
        $this->actingAs($this->user);
        $response = $this->put(route('midias.update', $midia), [
            'title' => 'Updated by different user',
            'type' => 'image',
            'empresa_id' => $this->empresa->id,
        ]);

        $midia->refresh();
        
        $this->assertEquals($otherUser->id, $midia->created_by); // Não deve mudar
        $this->assertEquals($this->user->id, $midia->updated_by); // Deve ser o usuário atual
    }

    public function test_userstamps_are_recorded_on_delete()
    {
        $midia = Midia::factory()->create(['empresa_id' => $this->empresa->id]);

        $response = $this->delete(route('midias.destroy', $midia));

        $midia->refresh();
        
        $this->assertEquals($this->user->id, $midia->deleted_by);
        $this->assertNotNull($midia->deleted_at);
    }
}