<?php

namespace Tests\Feature\Publicacao;

use Tests\TestCase;
use App\Models\{Publicacao, Criativo, Empresa, User};
use Illuminate\Foundation\Testing\RefreshDatabase;

class PublicacaoCrudTest extends TestCase
{
    use RefreshDatabase;

    protected $user;
    protected $franquia;
    protected $cliente;
    protected $criativo;

    protected function setUp(): void
    {
        parent::setUp();

        // Desabilitar todos middlewares de autenticação para testes
        $this->withoutMiddleware();

        $this->user = User::factory()->create();
        $this->franquia = Empresa::factory()->create(['tipo_empresa' => 'franquia']);
        $this->cliente = Empresa::factory()->create(['tipo_empresa' => 'cliente']);
        $this->criativo = Criativo::factory()->create([
            'status_aprovacao' => 'aprovado',
            'status' => 0
        ]);
    }

    public function test_usuario_autenticado_pode_acessar_listagem_de_publicacoes()
    {
        $response = $this->actingAs($this->user)->get(route('publicacoes.index'));

        $response->assertStatus(200);
        $response->assertViewIs('publicacoes.index');
        $response->assertViewHas(['publicacoes', 'stats', 'franquias', 'clientes']);
    }

    public function test_usuario_nao_autenticado_e_redirecionado_para_login()
    {
        $response = $this->get(route('publicacoes.index'));

        $response->assertRedirect(route('login'));
    }

    public function test_listagem_exibe_publicacoes_paginadas()
    {
        Publicacao::factory()->count(20)->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $response = $this->actingAs($this->user)->get(route('publicacoes.index'));

        $response->assertStatus(200);
    }

    public function test_usuario_pode_acessar_formulario_de_criacao()
    {
        $response = $this->actingAs($this->user)->get(route('publicacoes.create'));

        $response->assertStatus(200);
        $response->assertViewIs('publicacoes.create');
        $response->assertViewHas(['criativos', 'franquias', 'clientes', 'empresas']);
    }

    public function test_usuario_pode_criar_publicacao_com_dados_validos()
    {
        $data = [
            'criativo_id' => $this->criativo->id,
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
            'titulo' => 'Publicação Teste',
            'data_inicio' => now()->format('Y-m-d'),
            'data_fim' => now()->addDays(30)->format('Y-m-d'),
            'link_destino' => 'https://example.com',
            'habilitar_clique' => true,
            'track_impressoes' => true,
            'track_cliques' => true,
            'valor_contrato' => 1500.00,
            'impressoes_contratadas' => 10000,
            'posicao' => 1,
            'status' => 'ativa',
            'tipo_alcance' => 'todos',
        ];

        $response = $this->actingAs($this->user)->post(route('publicacoes.store'), $data);

        $response->assertRedirect(route('publicacoes.index'));
        $response->assertSessionHas('success');

        $this->assertDatabaseHas('publicacoes', [
            'titulo' => 'Publicação Teste',
            'criativo_id' => $this->criativo->id,
        ]);
    }

    public function test_criacao_falha_com_dados_invalidos()
    {
        $data = [
            'titulo' => '',
            'data_inicio' => '',
        ];

        $response = $this->actingAs($this->user)->post(route('publicacoes.store'), $data);

        $response->assertSessionHasErrors(['titulo', 'data_inicio']);
    }

    public function test_usuario_pode_visualizar_detalhes_da_publicacao()
    {
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $response = $this->actingAs($this->user)->get(route('publicacoes.show', $publicacao));

        $response->assertStatus(200);
        $response->assertViewIs('publicacoes.show');
        $response->assertViewHas(['publicacao', 'stats']);
    }

    public function test_usuario_pode_acessar_formulario_de_edicao()
    {
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $response = $this->actingAs($this->user)->get(route('publicacoes.edit', $publicacao));

        $response->assertStatus(200);
        $response->assertViewIs('publicacoes.edit');
        $response->assertViewHas(['publicacao', 'criativos', 'franquias', 'clientes']);
    }

    public function test_usuario_pode_atualizar_publicacao()
    {
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
            'titulo' => 'Título Original',
        ]);

        $data = [
            'criativo_id' => $publicacao->criativo_id,
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
            'titulo' => 'Título Atualizado',
            'data_inicio' => $publicacao->data_inicio->format('Y-m-d'),
            'data_fim' => $publicacao->data_fim->format('Y-m-d'),
            'posicao' => 1,
            'status' => 'ativa',
            'tipo_alcance' => 'todos',
        ];

        $response = $this->actingAs($this->user)->put(route('publicacoes.update', $publicacao), $data);

        $response->assertRedirect(route('publicacoes.index'));
        $this->assertDatabaseHas('publicacoes', [
            'id' => $publicacao->id,
            'titulo' => 'Título Atualizado',
        ]);
    }

    public function test_usuario_pode_deletar_publicacao()
    {
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $response = $this->actingAs($this->user)->delete(route('publicacoes.destroy', $publicacao));

        $response->assertRedirect(route('publicacoes.index'));
        $this->assertSoftDeleted('publicacoes', ['id' => $publicacao->id]);
    }

    public function test_usuario_pode_pausar_publicacao_ativa()
    {
        $publicacao = Publicacao::factory()->create([
            'status' => 'ativa',
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $response = $this->actingAs($this->user)
            ->patch(route('publicacoes.toggleStatus', $publicacao));

        $response->assertRedirect(route('publicacoes.index'));
        $publicacao->refresh();
        $this->assertEquals('pausada', $publicacao->status);
    }
}
