use App\Models\Publicacao;
use App\Models\User;
use App\Models\Criativo;
use App\Models\Empresa;
use Illuminate\Foundation\Testing\RefreshDatabase;

uses(RefreshDatabase::class);

test('it can display a list of publicacoes', function () {
    // Criar um usuário administrador para autenticação nos testes
    $adminUser = User::factory()->create([
        'name' => 'Super Admin',
        'email' => 'admin@example.com',
        'password' => bcrypt('password'),
        'is_superadmin' => true,
        'status_cadastro' => 'aprovado',
    ]);

    // Criar algumas publicações para testar a listagem
    $publicacoes = Publicacao::factory()->count(3)->vigente()->create();

    // Autenticar como administrador
    $this->actingAs($adminUser);

    // Fazer uma requisição GET para a rota de índice de publicações
    $response = $this->get(route('publicacoes.index'));

    // Verificar se a requisição foi bem-sucedida (status 200)
    $response->assertStatus(200);

    // Verificar se as publicações criadas estão visíveis na página
    $response->assertViewHas('publicacoes');
    foreach ($publicacoes as $publicacao) {
        $response->assertSee($publicacao->titulo);
    }
});

test('it can display the create publicacao form', function () {
    $adminUser = User::factory()->create([
        'name' => 'Super Admin',
        'email' => 'admin@example.com',
        'password' => bcrypt('password'),
        'is_superadmin' => true,
        'status_cadastro' => 'aprovado',
    ]);
    $this->actingAs($adminUser);

    $response = $this->get(route('publicacoes.create'));

    $response->assertStatus(200);
    $response->assertViewIs('publicacoes.create');
});

test('it can create a publicacao', function () {
    $adminUser = User::factory()->create([
        'name' => 'Super Admin',
        'email' => 'admin@example.com',
        'password' => bcrypt('password'),
        'is_superadmin' => true,
        'status_cadastro' => 'aprovado',
    ]);
    $this->actingAs($adminUser);

    $criativo = Criativo::factory()->aprovado()->create();
    $franquia = Empresa::factory()->create(['tipo_empresa' => 'franquia']);
    $cliente = Empresa::factory()->create(['tipo_empresa' => 'cliente']);

    $publicacaoData = [
        'criativo_id' => $criativo->id,
        'franquia_id' => $franquia->id,
        'cliente_id' => $cliente->id,
        'titulo' => 'Nova Publicação de Teste',
        'data_inicio' => now()->format('Y-m-d'),
        'data_fim' => now()->addDays(7)->format('Y-m-d'),
        'link_destino' => 'http://example.com',
        'habilitar_clique' => true,
        'track_impressoes' => true,
        'track_cliques' => true,
        'valor_contrato' => 1500.00,
        'impressoes_contratadas' => 100000,
        'posicao' => 1,
        'status' => 'ativa',
        'tipo_alcance' => 'todos',
    ];

    $response = $this->post(route('publicacoes.store'), $publicacaoData)->withoutMiddleware(\App\Http\Middleware\VerifyCsrfToken::class);

    $response->assertStatus(302);
    $response->assertRedirect(route('publicacoes.index'));
    $this->assertDatabaseHas('publicacoes', ['titulo' => 'Nova Publicação de Teste']);
    $response->assertSessionHas('success', 'Publicação criada com sucesso!');
});

test('it cannot create a publicacao with invalid data', function () {
    $adminUser = User::factory()->create([
        'name' => 'Super Admin',
        'email' => 'admin@example.com',
        'password' => bcrypt('password'),
        'is_superadmin' => true,
        'status_cadastro' => 'aprovado',
    ]);
    $this->actingAs($adminUser);

    $publicacaoData = [
        'criativo_id' => 9999, // ID inválido
        'titulo' => '', // Título vazio
        'data_fim' => now()->subDays(1)->format('Y-m-d'), // Data fim antes da data início
    ];

    $response = $this->post(route('publicacoes.store'), $publicacaoData)->withoutMiddleware(\App\Http\Middleware\VerifyCsrfToken::class);

    $response->assertStatus(302);
    $response->assertSessionHasErrors(['criativo_id', 'titulo', 'data_inicio', 'data_fim', 'franquia_id', 'cliente_id', 'posicao', 'status', 'tipo_alcance']);
    $this->assertDatabaseMissing('publicacoes', ['titulo' => '']);
});