<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use App\Models\User;
use App\Models\Group;
use App\Models\Permission;

class RbacFlowTest extends TestCase
{
    use RefreshDatabase;

    protected $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
        $this->actingAs($this->user);
    }

    // Permissions CRUD
    public function test_can_view_permissions_index()
    {
        $response = $this->get(route('permissions.index'));
        $response->assertStatus(200);
        $response->assertViewIs('permissions.index');
    }

    public function test_can_create_permission()
    {
        $permissionData = [
            'name' => 'Test Permission',
            'description' => 'A test permission',
            'subsystem' => 'Test',
            'action' => 'test',
        ];

        $response = $this->post(route('permissions.store'), $permissionData);

        $response->assertRedirect(route('permissions.index'));
        $this->assertDatabaseHas('permissions', ['slug' => 'Test.test']);
    }

    public function test_can_update_permission()
    {
        $permission = Permission::factory()->create();
        $updatedData = [
            'name' => 'Updated Permission',
            'description' => 'An updated description',
            'subsystem' => 'Test',
            'action' => 'updated',
        ];

        $response = $this->put(route('permissions.update', $permission), $updatedData);

        $response->assertRedirect(route('permissions.index'));
        $this->assertDatabaseHas('permissions', ['id' => $permission->id, 'slug' => 'Test.updated']);
    }

    public function test_can_delete_permission()
    {
        $permission = Permission::factory()->create();

        $response = $this->delete(route('permissions.destroy', $permission));

        $response->assertRedirect(route('permissions.index'));
        $this->assertDatabaseMissing('permissions', ['id' => $permission->id]);
    }

    // Groups CRUD
    public function test_can_view_groups_index()
    {
        $response = $this->get(route('groups.index'));
        $response->assertStatus(200);
        $response->assertViewIs('groups.index');
    }

    public function test_can_create_group()
    {
        $groupData = [
            'name' => 'Test Group',
            'description' => 'A test group',
        ];

        $response = $this->post(route('groups.store'), $groupData);

        $response->assertRedirect(route('groups.index'));
        $this->assertDatabaseHas('groups', $groupData);
    }

    public function test_can_update_group()
    {
        $group = Group::factory()->create();
        $updatedData = [
            'name' => 'Updated Group',
            'description' => 'An updated description',
        ];

        $response = $this->put(route('groups.update', $group), $updatedData);

        $response->assertRedirect(route('groups.index'));
        $this->assertDatabaseHas('groups', ['id' => $group->id, 'name' => 'Updated Group']);
    }

    public function test_can_delete_group()
    {
        $group = Group::factory()->create();

        $response = $this->delete(route('groups.destroy', $group));

        $response->assertRedirect(route('groups.index'));
        $this->assertDatabaseMissing('groups', ['id' => $group->id]);
    }

    // Permission Matrix
    public function test_can_view_group_permissions_matrix()
    {
        $group = Group::factory()->create();
        $response = $this->get(route('groups.permissions.edit', $group));
        $response->assertStatus(200);
        $response->assertViewIs('groups.permissions.matrix');
    }

    public function test_can_update_group_permissions()
    {
        $group = Group::factory()->create();
        $permissions = Permission::factory(3)->create();
        $permissionIds = $permissions->pluck('id')->toArray();

        $response = $this->post(route('groups.permissions.update', $group), [
            'permissions' => $permissionIds,
        ]);

        $response->assertRedirect(route('groups.index'));
        $this->assertCount(3, $group->refresh()->permissions);
        $this->assertDatabaseHas('group_permission', ['group_id' => $group->id, 'permission_id' => $permissionIds[0]]);
        $this->assertDatabaseHas('group_permission', ['group_id' => $group->id, 'permission_id' => $permissionIds[1]]);
        $this->assertDatabaseHas('group_permission', ['group_id' => $group->id, 'permission_id' => $permissionIds[2]]);
    }
}
