<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use App\Models\User;

class UserCrudTest extends TestCase
{
    use RefreshDatabase;

    protected $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create([
            'is_superadmin' => true,
            'status_cadastro' => 'aprovado',
        ]);
        $this->actingAs($this->user);
    }

    public function test_can_view_users_index()
    {
        $response = $this->get(route('users.index'));
        $response->assertStatus(200);
        $response->assertViewIs('users.index');
    }

    public function test_can_view_user_create_form()
    {
        $response = $this->get(route('users.create'));
        $response->assertStatus(200);
        $response->assertViewIs('users.create');
    }

    public function test_can_create_user()
    {
        $userData = [
            'name' => 'New User',
            'email' => 'newuser@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ];

        $response = $this->post(route('users.store'), $userData);

        $response->assertRedirect(route('users.index'));
        $this->assertDatabaseHas('users', ['email' => 'newuser@example.com']);
    }

    public function test_can_view_user_edit_form()
    {
        $userToEdit = User::factory()->create();
        $response = $this->get(route('users.edit', $userToEdit));
        $response->assertStatus(200);
        $response->assertViewIs('users.edit');
    }

    public function test_can_update_user()
    {
        $userToUpdate = User::factory()->create();
        $updatedData = [
            'name' => 'Updated User Name',
            'email' => 'updateduser@example.com',
        ];

        $response = $this->put(route('users.update', $userToUpdate), $updatedData);

        $response->assertRedirect(route('users.index'));
        $this->assertDatabaseHas('users', ['id' => $userToUpdate->id, 'name' => 'Updated User Name']);
    }

    public function test_can_delete_user()
    {
        $userToDelete = User::factory()->create();
        $response = $this->delete(route('users.destroy', $userToDelete));

        $response->assertRedirect(route('users.index'));
        $this->assertDatabaseMissing('users', ['id' => $userToDelete->id]);
    }
}
