<?php

namespace Tests\Unit;

use App\Models\Publicacao;
use App\Models\AuditoriaSplash;
use Carbon\Carbon;
use Tests\TestCase;

class PublicacaoEstatisticasTest extends TestCase
{
    /**
     * Teste: Verificar se método estatisticas carrega dados corretamente
     */
    public function test_estatisticas_loads_auditoria_splash_data()
    {
        // Criar publicação de teste
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        // Criar dados de auditoria_splash
        $this->createAuditoriaSplashData($publicacao, 5);

        // Chamar o método (simulando)
        $response = $this->get(route('publicacoes.estatisticas', $publicacao));

        $response->assertStatus(200);
        $response->assertViewHas('stats');
        $response->assertViewHas('publicacao');
    }

    /**
     * Teste: Verificar cálculo de usuários únicos
     */
    public function test_total_usuarios_unicos_is_calculated_correctly()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        // Criar múltiplos registros com MACs diferentes
        AuditoriaSplash::create([
            'hs_serial' => 'HS001',
            'cl_mac_address' => 'AA:BB:CC:DD:EE:01',
            'pag_click' => 'http://example.com',
            'created_at' => now(),
        ]);

        AuditoriaSplash::create([
            'hs_serial' => 'HS001',
            'cl_mac_address' => 'AA:BB:CC:DD:EE:02',
            'pag_click' => 'http://example.com',
            'created_at' => now(),
        ]);

        AuditoriaSplash::create([
            'hs_serial' => 'HS001',
            'cl_mac_address' => 'AA:BB:CC:DD:EE:01', // MAC duplicado
            'pag_click' => 'http://example.com',
            'created_at' => now(),
        ]);

        $response = $this->get(route('publicacoes.estatisticas', $publicacao));
        
        $this->assertEquals(2, $response->viewData('stats')['total_usuarios_logs']);
    }

    /**
     * Teste: Verificar se top hotspots são ordenados corretamente
     */
    public function test_top_hotspots_are_sorted_by_impressions()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        // Criar dados com diferentes hotspots
        for ($i = 1; $i <= 5; $i++) {
            for ($j = 1; $j <= $i * 2; $j++) {
                AuditoriaSplash::create([
                    'hs_serial' => "HS{$i}",
                    'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($j, 2, '0', STR_PAD_LEFT),
                    'pag_click' => 'http://example.com',
                    'created_at' => now(),
                ]);
            }
        }

        $response = $this->get(route('publicacoes.estatisticas', $publicacao));
        $stats = $response->viewData('stats');

        // Verificar ordenação (primeiro deve ter mais impressões)
        $this->assertGreaterThanOrEqual(
            $stats['top_hotspots'][1]->total_impressoes ?? 0,
            $stats['top_hotspots'][0]->total_impressoes
        );
    }

    /**
     * Teste: Verificar cálculo de dados por dia
     */
    public function test_impressoes_por_dia_calculated_correctly()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        $dataOntem = Carbon::now()->subDay()->startOfDay();
        $dataHoje = Carbon::now()->startOfDay();

        // Criar registros para ontem
        for ($i = 0; $i < 3; $i++) {
            AuditoriaSplash::create([
                'hs_serial' => 'HS001',
                'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($i, 2, '0', STR_PAD_LEFT),
                'pag_click' => 'http://example.com',
                'created_at' => $dataOntem->copy()->addHours(10),
            ]);
        }

        // Criar registros para hoje
        for ($i = 0; $i < 5; $i++) {
            AuditoriaSplash::create([
                'hs_serial' => 'HS001',
                'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($i + 10, 2, '0', STR_PAD_LEFT),
                'pag_click' => 'http://example.com',
                'created_at' => $dataHoje->copy()->addHours(10),
            ]);
        }

        $response = $this->get(route('publicacoes.estatisticas', $publicacao));
        $stats = $response->viewData('stats');

        $this->assertGreaterThan(0, count($stats['impressoes_por_dia']));
    }

    /**
     * Teste: Verificar se MAC é formatado corretamente
     */
    public function test_mac_address_formatting()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        AuditoriaSplash::create([
            'hs_serial' => 'HS001',
            'cl_mac_address' => 'AA:BB:CC:DD:EE:FF',
            'pag_click' => 'http://example.com',
            'created_at' => now(),
        ]);

        $response = $this->get(route('publicacoes.estatisticas', $publicacao));
        $stats = $response->viewData('stats');

        // Verificar se MAC está formatado no logs_acesso
        $this->assertStringContainsString('***', $stats['logs_acesso'][0]['mac_parcial']);
    }

    /**
     * Teste: Verificar CTR (sempre 100% para splash)
     */
    public function test_ctr_is_always_100_percent_for_splash()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        $this->createAuditoriaSplashData($publicacao, 10);

        $response = $this->get(route('publicacoes.estatisticas', $publicacao));
        $stats = $response->viewData('stats');

        $this->assertEquals(100, $stats['ctr']);
    }

    /**
     * Helper: Criar dados de auditoria_splash
     */
    private function createAuditoriaSplashData($publicacao, $count)
    {
        for ($i = 0; $i < $count; $i++) {
            AuditoriaSplash::create([
                'hs_serial' => 'HS' . str_pad($i % 3 + 1, 3, '0', STR_PAD_LEFT),
                'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($i, 2, '0', STR_PAD_LEFT),
                'pag_click' => 'http://example.com/page-' . $i,
                'imagem' => 'criativo_' . $i . '.jpg',
                'dsc_identity' => 'Equipment-' . ($i % 2 + 1),
                'created_at' => now()->subDays(rand(0, 9)),
            ]);
        }
    }
}
