#!/usr/bin/env php
<?php

/**
 * Envio de Email de Teste (SEM BANCO DE DADOS)
 * =============================================
 * Envia email de demonstração usando dados fictícios
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\{User, Empresa};
use App\Mail\{CadastroAprovado, CadastroRejeitado, NovoCadastroPendente};
use Illuminate\Support\Facades\Mail;

echo "\n";
echo "╔═══════════════════════════════════════════════════════════╗\n";
echo "║       ENVIO DE EMAIL DE TESTE - FreeFi Admin             ║\n";
echo "╚═══════════════════════════════════════════════════════════╝\n";
echo "\n";

// Email de destino
$emailDestino = 'linuxewireless@gmail.com';

// Criar usuário fictício (não salva no banco)
$empresa = new Empresa();
$empresa->id = 1;
$empresa->name = 'FreeFi Franquia São Paulo';
$empresa->email = 'contato@freefi-sp.com.br';
$empresa->tipo_empresa = 'franquia';

$user = new User();
$user->id = 999;
$user->name = 'João Silva';
$user->email = $emailDestino;
$user->status_cadastro = 'pendente';
$user->phone = '(11) 98765-4321';
$user->created_at = now();
$user->setRelation('empresa', $empresa);

echo "📊 DADOS FICTÍCIOS PARA DEMONSTRAÇÃO:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "  Nome:    {$user->name}\n";
echo "  Email:   {$user->email}\n";
echo "  Empresa: {$empresa->name}\n";
echo "  Tipo:    {$empresa->tipo_empresa}\n";
echo "  Status:  {$user->status_cadastro}\n";
echo "\n";

echo "📧 ESCOLHA O TIPO DE EMAIL PARA ENVIAR:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "  [1] ✅ Cadastro Aprovado (verde)\n";
echo "  [2] ❌ Cadastro Rejeitado (vermelho)\n";
echo "  [3] 🔔 Novo Cadastro Pendente (laranja)\n";
echo "  [4] 📤 Enviar TODOS os tipos (recomendado)\n";
echo "\n";
echo "Digite sua escolha (1-4): ";

$escolha = trim(fgets(STDIN));

echo "\n";
echo "📤 Enviando para: $emailDestino\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$enviados = 0;

try {
    switch ($escolha) {
        case '1':
            echo "📧 Enviando: Cadastro Aprovado...\n";
            // Criar mailable sem ShouldQueue
            $mailable = new CadastroAprovado($user);
            Mail::to($emailDestino)->send($mailable);
            echo "   ✅ Enviado com sucesso!\n";
            echo "   Assunto: ✅ Cadastro Aprovado - FreeFi Admin\n";
            $enviados++;
            break;
            
        case '2':
            echo "📧 Enviando: Cadastro Rejeitado...\n";
            $motivo = "Documentos incompletos. Por favor, entre em contato com o suporte para mais informações.";
            $mailable = new CadastroRejeitado($user, $motivo);
            Mail::to($emailDestino)->send($mailable);
            echo "   ✅ Enviado com sucesso!\n";
            echo "   Assunto: ❌ Cadastro Rejeitado - FreeFi Admin\n";
            echo "   Motivo incluído: Sim\n";
            $enviados++;
            break;
            
        case '3':
            echo "📧 Enviando: Novo Cadastro Pendente...\n";
            $mailable = new NovoCadastroPendente($user);
            Mail::to($emailDestino)->send($mailable);
            echo "   ✅ Enviado com sucesso!\n";
            echo "   Assunto: 🔔 Novo Cadastro Pendente de Aprovação - FreeFi Admin\n";
            $enviados++;
            break;
            
        case '4':
            echo "📧 [1/3] Enviando: Cadastro Aprovado...\n";
            $mailable1 = new class($user) extends CadastroAprovado {
                // Remove ShouldQueue para envio direto
            };
            Mail::to($emailDestino)->send(new CadastroAprovado($user));
            echo "      ✅ Enviado!\n\n";
            $enviados++;
            
            sleep(2); // Pequeno delay entre envios
            
            echo "📧 [2/3] Enviando: Cadastro Rejeitado...\n";
            $motivo = "Documentos incompletos. Por favor, entre em contato com o suporte para mais informações.";
            Mail::to($emailDestino)->send(new CadastroRejeitado($user, $motivo));
            echo "      ✅ Enviado!\n\n";
            $enviados++;
            
            sleep(2);
            
            echo "📧 [3/3] Enviando: Novo Cadastro Pendente...\n";
            Mail::to($emailDestino)->send(new NovoCadastroPendente($user));
            echo "      ✅ Enviado!\n\n";
            $enviados++;
            break;
            
        default:
            echo "❌ Opção inválida!\n";
            exit(1);
    }
    
    echo "\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "✅ ENVIO CONCLUÍDO COM SUCESSO!\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    echo "📊 RESUMO:\n";
    echo "   • Emails enviados: $enviados\n";
    echo "   • Destinatário: $emailDestino\n";
    echo "   • Servidor SMTP: " . config('mail.mailers.smtp.host') . "\n";
    echo "   • Porta: " . config('mail.mailers.smtp.port') . "\n";
    echo "   • Remetente: " . config('mail.from.address') . "\n\n";
    
    echo "📋 PRÓXIMOS PASSOS:\n";
    echo "   1. Verifique a caixa de entrada de: $emailDestino\n";
    echo "   2. Verifique também a pasta de SPAM/Lixo Eletrônico\n";
    echo "   3. Os emails devem chegar em alguns segundos/minutos\n";
    echo "   4. Observe o design profissional com:\n";
    echo "      • Gradientes coloridos\n";
    echo "      • Ícones grandes (✅ ❌ 🔔)\n";
    echo "      • Botões de ação\n";
    echo "      • CSS inline responsivo\n\n";
    
    if ($escolha == '4') {
        echo "💡 COMPARE OS 3 DESIGNS:\n";
        echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
        echo "   ✅ Email VERDE (Cadastro Aprovado)\n";
        echo "      • Header verde com gradiente\n";
        echo "      • Tom positivo e motivador\n";
        echo "      • Botão 'Acessar Sistema'\n";
        echo "      • Dica de segurança\n\n";
        
        echo "   ❌ Email VERMELHO (Cadastro Rejeitado)\n";
        echo "      • Header vermelho com gradiente\n";
        echo "      • Tom empático\n";
        echo "      • Motivo da rejeição em destaque\n";
        echo "      • Lista de próximos passos\n\n";
        
        echo "   🔔 Email LARANJA (Novo Cadastro Pendente)\n";
        echo "      • Header laranja com gradiente\n";
        echo "      • Badge 'REQUER AÇÃO IMEDIATA'\n";
        echo "      • Dados completos do usuário\n";
        echo "      • Badge colorido do tipo de empresa\n";
        echo "      • Botão 'Gerenciar Cadastro'\n\n";
    }
    
    echo "🎨 CARACTERÍSTICAS DOS EMAILS:\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "   ✓ HTML5 completo com DOCTYPE\n";
    echo "   ✓ CSS inline (compatível com Gmail, Outlook, etc.)\n";
    echo "   ✓ Design responsivo (mobile-friendly)\n";
    echo "   ✓ Gradientes e sombras modernas\n";
    echo "   ✓ Ícones e emojis\n";
    echo "   ✓ Botões CTA com hover effects\n";
    echo "   ✓ Footer com informações da empresa\n\n";
    
    exit(0);
    
} catch (\Exception $e) {
    echo "\n";
    echo "❌ ERRO AO ENVIAR EMAIL!\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "Detalhes: " . $e->getMessage() . "\n";
    echo "\n";
    echo "Stack trace:\n";
    echo $e->getTraceAsString() . "\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    echo "🔧 VERIFICAÇÕES:\n";
    echo "   • Configuração SMTP no .env está correta?\n";
    echo "   • Servidor SMTP está acessível?\n";
    echo "   • Credenciais estão corretas?\n";
    echo "   • Execute: php test-smtp-connection.php\n\n";
    
    exit(1);
}
