<?php

/**
 * Teste - Botões de Aprovação/Rejeição de Usuários
 * Valida funcionalidade de moderação de cadastros
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(\Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Models\Empresa;

class TestOutput {
    public static function success($msg) { echo "\033[32m✓\033[0m {$msg}\n"; }
    public static function error($msg) { echo "\033[31m✗\033[0m {$msg}\n"; }
    public static function info($msg) { echo "\033[36mℹ\033[0m {$msg}\n"; }
    public static function title($msg) { echo "\n\033[1;33m{$msg}\033[0m\n" . str_repeat('═', 70) . "\n"; }
}

$passed = 0;
$failed = 0;

echo "\033[1m\n╔══════════════════════════════════════════════════════════════════╗\n";
echo "║      TESTE - SISTEMA DE APROVAÇÃO/REJEIÇÃO DE CADASTROS         ║\n";
echo "╚══════════════════════════════════════════════════════════════════╝\033[0m\n";

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('PREPARAÇÃO: Criar Usuário de Teste');

DB::beginTransaction();

$franquia = Empresa::where('tipo_empresa', 'franquia')
    ->where('status', 0)
    ->first();

if (!$franquia) {
    TestOutput::error("Nenhuma franquia ativa encontrada!");
    exit(1);
}

TestOutput::success("Franquia encontrada: {$franquia->name}");

// Criar usuário pendente
$testUser = User::create([
    'name' => 'Teste Aprovação ' . time(),
    'email' => 'teste.aprovacao.' . time() . '@test.com',
    'password' => bcrypt('senha123'),
    'empresa_id' => $franquia->id,
    'status_cadastro' => 'pendente',
]);

TestOutput::success("Usuário de teste criado (ID: {$testUser->id})");
TestOutput::info("  → Status inicial: {$testUser->status_cadastro}");
$passed++;

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 1: Verificar Rotas de Aprovação/Rejeição');

$routes = [
    'users.approve' => 'Rota de aprovar',
    'users.reject' => 'Rota de rejeitar',
];

foreach ($routes as $name => $description) {
    if (\Illuminate\Support\Facades\Route::has($name)) {
        TestOutput::success("{$description} existe");
        $passed++;
    } else {
        TestOutput::error("{$description} NÃO existe!");
        $failed++;
    }
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 2: Aprovar Usuário (pendente → aprovado)');

$testUser->update(['status_cadastro' => 'aprovado']);
$testUser->refresh();

if ($testUser->status_cadastro === 'aprovado') {
    TestOutput::success("Status alterado para 'aprovado'");
    TestOutput::info("  → Status anterior: pendente");
    TestOutput::info("  → Status atual: {$testUser->status_cadastro}");
    $passed++;
} else {
    TestOutput::error("Falha ao aprovar usuário!");
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 3: Rejeitar Usuário (aprovado → rejeitado)');

$testUser->update(['status_cadastro' => 'rejeitado']);
$testUser->refresh();

if ($testUser->status_cadastro === 'rejeitado') {
    TestOutput::success("Status alterado para 'rejeitado'");
    TestOutput::info("  → Status anterior: aprovado");
    TestOutput::info("  → Status atual: {$testUser->status_cadastro}");
    $passed++;
} else {
    TestOutput::error("Falha ao rejeitar usuário!");
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 4: Aprovar Novamente (rejeitado → aprovado)');

$testUser->update(['status_cadastro' => 'aprovado']);
$testUser->refresh();

if ($testUser->status_cadastro === 'aprovado') {
    TestOutput::success("Usuário rejeitado pode ser aprovado novamente");
    TestOutput::info("  → Status anterior: rejeitado");
    TestOutput::info("  → Status atual: {$testUser->status_cadastro}");
    $passed++;
} else {
    TestOutput::error("Falha ao re-aprovar usuário!");
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 5: Verificar View com Botões');

$viewPath = resource_path('views/users/index.blade.php');

if (file_exists($viewPath)) {
    $content = file_get_contents($viewPath);
    
    $checks = [
        'approve-user' => 'Classe do botão aprovar',
        'reject-user' => 'Classe do botão rejeitar',
        'ph-check-circle' => 'Ícone de aprovação',
        'ph-x-circle' => 'Ícone de rejeição',
        'approve-form-' => 'Form de aprovação',
        'reject-form-' => 'Form de rejeição',
        'users.approve' => 'Route de aprovação',
        'users.reject' => 'Route de rejeição',
        'Aprovar Cadastro' => 'Tooltip aprovar',
        'Rejeitar Cadastro' => 'Tooltip rejeitar',
    ];
    
    foreach ($checks as $needle => $description) {
        if (strpos($content, $needle) !== false) {
            TestOutput::success($description);
            $passed++;
        } else {
            TestOutput::error("{$description} - NÃO ENCONTRADO!");
            $failed++;
        }
    }
    
} else {
    TestOutput::error("View não encontrada!");
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 6: Verificar Controller Methods');

$controllerPath = app_path('Http/Controllers/UserController.php');

if (file_exists($controllerPath)) {
    $content = file_get_contents($controllerPath);
    
    $checks = [
        'public function approve' => 'Método approve()',
        'public function reject' => 'Método reject()',
        'status_cadastro' => 'Uso do campo status_cadastro',
        'aprovado' => 'Status aprovado',
        'rejeitado' => 'Status rejeitado',
    ];
    
    foreach ($checks as $needle => $description) {
        if (strpos($content, $needle) !== false) {
            TestOutput::success($description);
            $passed++;
        } else {
            TestOutput::error("{$description} - NÃO ENCONTRADO!");
            $failed++;
        }
    }
    
} else {
    TestOutput::error("Controller não encontrado!");
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 7: Estatísticas de Usuários por Status');

$stats = [
    'pendente' => User::where('status_cadastro', 'pendente')->count(),
    'aprovado' => User::where('status_cadastro', 'aprovado')->count(),
    'rejeitado' => User::where('status_cadastro', 'rejeitado')->count(),
];

TestOutput::success("Estatísticas coletadas:");
TestOutput::info("  → Pendentes: {$stats['pendente']}");
TestOutput::info("  → Aprovados: {$stats['aprovado']}");
TestOutput::info("  → Rejeitados: {$stats['rejeitado']}");
TestOutput::info("  → Total: " . array_sum($stats));
$passed++;

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('LIMPEZA: Remover Usuário de Teste');

DB::rollBack();

$userExists = User::where('email', $testUser->email)->exists();

if (!$userExists) {
    TestOutput::success("Transação revertida - usuário removido");
    $passed++;
} else {
    TestOutput::error("Usuário ainda existe!");
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
echo "\n\033[1m╔══════════════════════════════════════════════════════════════════╗\n";
echo "║                        RESUMO DO TESTE                           ║\n";
echo "╚══════════════════════════════════════════════════════════════════╝\033[0m\n\n";

echo "Total de testes: \033[1m" . ($passed + $failed) . "\033[0m\n";
echo "\033[32m✓ Passados: {$passed}\033[0m\n";

if ($failed > 0) {
    echo "\033[31m✗ Falhados: {$failed}\033[0m\n";
}

$percentage = ($passed + $failed) > 0 ? round(($passed / ($passed + $failed)) * 100, 2) : 0;

echo "\n\033[1mTaxa de sucesso: ";
if ($percentage === 100.0) {
    echo "\033[32m{$percentage}%\033[0m ✓\n\n";
} else {
    echo "\033[33m{$percentage}%\033[0m ⚠️\n\n";
}

echo "\033[36m══════════════════════════════════════════════════════════════════\033[0m\n";
echo "\033[1mFUNCIONALIDADES IMPLEMENTADAS:\033[0m\n";
echo "  ✓ Botão 'Aprovar' para usuários pendentes/rejeitados\n";
echo "  ✓ Botão 'Rejeitar' para usuários pendentes\n";
echo "  ✓ Confirmação via JavaScript\n";
echo "  ✓ Métodos approve() e reject() no Controller\n";
echo "  ✓ Rotas users.approve e users.reject\n";
echo "  ✓ Badges coloridos (verde/amarelo/vermelho)\n";
echo "  ✓ Mensagens de feedback (success/warning/info)\n";
echo "  ✓ Preparado para envio de emails (TODO)\n";
echo "\033[36m══════════════════════════════════════════════════════════════════\033[0m\n\n";

echo "\033[1;33mESTATÍSTICAS ATUAIS:\033[0m\n";
echo "  🟡 Pendentes: {$stats['pendente']}\n";
echo "  🟢 Aprovados: {$stats['aprovado']}\n";
echo "  🔴 Rejeitados: {$stats['rejeitado']}\n";
echo "  📊 Total: " . array_sum($stats) . "\n\n";

exit($failed > 0 ? 1 : 0);
