<?php

/**
 * Teste - Botão de Limpar Filtros
 * Verifica se o botão aparece corretamente quando há filtros ativos
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(\Illuminate\Contracts\Http\Kernel::class);
$kernel->bootstrap();

use Illuminate\Http\Request;
use App\Models\User;

class TestOutput {
    public static function success($msg) { echo "\033[32m✓\033[0m {$msg}\n"; }
    public static function error($msg) { echo "\033[31m✗\033[0m {$msg}\n"; }
    public static function info($msg) { echo "\033[36mℹ\033[0m {$msg}\n"; }
    public static function title($msg) { echo "\n\033[1;33m{$msg}\033[0m\n" . str_repeat('═', 70) . "\n"; }
}

$passed = 0;
$failed = 0;

echo "\033[1m\n╔══════════════════════════════════════════════════════════════════╗\n";
echo "║           TESTE - BOTÃO DE LIMPAR FILTROS (X)                   ║\n";
echo "╚══════════════════════════════════════════════════════════════════╝\033[0m\n";

// Autenticar
$user = User::where('status_cadastro', 'aprovado')->first();
if (!$user) {
    $user = User::first();
}
auth()->loginUsingId($user->id);

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 1: Usuários - Sem Filtro (Botão NÃO deve aparecer)');

$request = Request::create('/users', 'GET');
$response = $kernel->handle($request);

if ($response->getStatusCode() === 200) {
    $content = $response->getContent();
    
    // Verificar se o campo de busca existe
    if (strpos($content, 'name="q"') !== false) {
        TestOutput::success("Campo de busca presente");
        $passed++;
    } else {
        TestOutput::error("Campo de busca não encontrado!");
        $failed++;
    }
    
    // Verificar se o botão de limpar NÃO aparece (sem filtro ativo)
    if (strpos($content, 'Limpar filtros') === false && strpos($content, 'ph-x') === false) {
        TestOutput::success("Botão de limpar NÃO aparece (correto - sem filtro)");
        $passed++;
    } else {
        TestOutput::error("Botão de limpar apareceu sem filtro ativo!");
        $failed++;
    }
    
} else {
    TestOutput::error("GET /users falhou com status " . $response->getStatusCode());
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 2: Usuários - Com Filtro (Botão DEVE aparecer)');

$request = Request::create('/users?q=teste', 'GET');
$response = $kernel->handle($request);

if ($response->getStatusCode() === 200) {
    $content = $response->getContent();
    
    // Verificar se o valor do filtro está presente
    if (strpos($content, 'value="teste"') !== false) {
        TestOutput::success("Valor do filtro presente no campo");
        $passed++;
    } else {
        TestOutput::error("Valor do filtro não encontrado!");
        $failed++;
    }
    
    // Verificar se o botão de limpar aparece
    if (strpos($content, 'Limpar filtros') !== false || strpos($content, 'ph-x') !== false) {
        TestOutput::success("Botão de limpar filtros APARECE (correto)");
        $passed++;
    } else {
        TestOutput::error("Botão de limpar não apareceu com filtro ativo!");
        $failed++;
    }
    
    // Verificar se o link aponta para a rota sem filtro
    if (strpos($content, 'href="' . route('users.index') . '"') !== false) {
        TestOutput::success("Link do botão aponta para rota sem filtro");
        $passed++;
    } else {
        TestOutput::error("Link do botão incorreto!");
        $failed++;
    }
    
} else {
    TestOutput::error("GET /users?q=teste falhou com status " . $response->getStatusCode());
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 3: Empresas - Verificar Consistência');

$request = Request::create('/empresas?q=freefi', 'GET');
$response = $kernel->handle($request);

if ($response->getStatusCode() === 200) {
    $content = $response->getContent();
    
    // Verificar estrutura do input-group
    if (strpos($content, 'input-group') !== false) {
        TestOutput::success("Estrutura input-group presente");
        $passed++;
    }
    
    // Verificar ícone de busca
    if (strpos($content, 'ph-magnifying-glass') !== false) {
        TestOutput::success("Ícone de lupa presente");
        $passed++;
    }
    
    // Verificar botão de limpar
    if (strpos($content, 'ph-x') !== false) {
        TestOutput::success("Botão X presente");
        $passed++;
    }
    
    // Verificar gap-2
    if (strpos($content, 'gap-2') !== false) {
        TestOutput::success("Espaçamento gap-2 presente");
        $passed++;
    }
    
} else {
    TestOutput::error("GET /empresas?q=freefi falhou");
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 4: Mídias - Verificar Consistência');

$request = Request::create('/midias?q=video', 'GET');
$response = $kernel->handle($request);

if ($response->getStatusCode() === 200) {
    $content = $response->getContent();
    
    // Verificar estrutura do input-group
    if (strpos($content, 'input-group') !== false) {
        TestOutput::success("Estrutura input-group presente");
        $passed++;
    }
    
    // Verificar ícone de busca
    if (strpos($content, 'ph-magnifying-glass') !== false) {
        TestOutput::success("Ícone de lupa presente");
        $passed++;
    }
    
    // Verificar botão de limpar
    if (strpos($content, 'ph-x') !== false) {
        TestOutput::success("Botão X presente");
        $passed++;
    }
    
    // Verificar ícone ph-plus nos botões
    if (strpos($content, 'ph-plus') !== false) {
        TestOutput::success("Ícone plus presente no botão");
        $passed++;
    }
    
} else {
    TestOutput::error("GET /midias?q=video falhou");
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════
TestOutput::title('TESTE 5: Verificar Views - Estrutura HTML');

$views = [
    'users' => resource_path('views/users/index.blade.php'),
    'empresas' => resource_path('views/empresas/index.blade.php'),
    'midias' => resource_path('views/midias/index.blade.php'),
];

foreach ($views as $name => $path) {
    if (file_exists($path)) {
        $content = file_get_contents($path);
        
        $checks = [
            'input-group' => 'Input group',
            'ph-magnifying-glass' => 'Ícone de lupa',
            'ph-x' => 'Ícone X',
            '@if(request(\'q\'))' => 'Condicional do botão',
            'Limpar filtros' => 'Texto do tooltip',
            'btn-outline-secondary' => 'Estilo do botão',
            'gap-2' => 'Espaçamento',
            'ph-plus' => 'Ícone plus',
        ];
        
        TestOutput::info("\nVerificando view: {$name}");
        
        $modulePass = 0;
        $moduleFail = 0;
        
        foreach ($checks as $needle => $description) {
            if (strpos($content, $needle) !== false) {
                echo "  \033[32m✓\033[0m {$description}\n";
                $modulePass++;
            } else {
                echo "  \033[31m✗\033[0m {$description}\n";
                $moduleFail++;
            }
        }
        
        if ($moduleFail === 0) {
            TestOutput::success("View {$name} 100% completa");
            $passed++;
        } else {
            TestOutput::error("View {$name} com {$moduleFail} itens faltando");
            $failed++;
        }
        
    } else {
        TestOutput::error("View {$name} não encontrada!");
        $failed++;
    }
}

// Desautenticar
auth()->logout();

// ══════════════════════════════════════════════════════════════════════
echo "\n\033[1m╔══════════════════════════════════════════════════════════════════╗\n";
echo "║                        RESUMO DO TESTE                           ║\n";
echo "╚══════════════════════════════════════════════════════════════════╝\033[0m\n\n";

echo "Total de testes: \033[1m" . ($passed + $failed) . "\033[0m\n";
echo "\033[32m✓ Passados: {$passed}\033[0m\n";

if ($failed > 0) {
    echo "\033[31m✗ Falhados: {$failed}\033[0m\n";
}

$percentage = ($passed + $failed) > 0 ? round(($passed / ($passed + $failed)) * 100, 2) : 0;

echo "\n\033[1mTaxa de sucesso: ";
if ($percentage === 100.0) {
    echo "\033[32m{$percentage}%\033[0m ✓\n\n";
} else {
    echo "\033[33m{$percentage}%\033[0m ⚠️\n\n";
}

echo "\033[36m══════════════════════════════════════════════════════════════════\033[0m\n";
echo "\033[1mFUNCIONALIDADE DO BOTÃO LIMPAR:\033[0m\n";
echo "  • Aparece SOMENTE quando há filtro ativo\n";
echo "  • Exibe ícone X (ph-x)\n";
echo "  • Tooltip: 'Limpar filtros'\n";
echo "  • Estilo: btn-outline-secondary\n";
echo "  • Ação: Remove todos os filtros da URL\n";
echo "  • Presente em: Usuários, Empresas, Mídias\n";
echo "\033[36m══════════════════════════════════════════════════════════════════\033[0m\n\n";

exit($failed > 0 ? 1 : 0);
