<?php

/**
 * Teste E2E - Requisições HTTP Reais
 * Simula requisições HTTP reais para encontrar erros de execução
 * que não aparecem em testes unitários/integração
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(\Illuminate\Contracts\Http\Kernel::class);

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Models\User;
use App\Models\Empresa;

// Cores
class TestOutput {
    public static function success($msg) { echo "\033[32m✓\033[0m {$msg}\n"; }
    public static function error($msg) { echo "\033[31m✗\033[0m {$msg}\n"; }
    public static function warning($msg) { echo "\033[33m⚠\033[0m {$msg}\n"; }
    public static function info($msg) { echo "\033[36mℹ\033[0m {$msg}\n"; }
    public static function title($msg) { echo "\n\033[1;33m{$msg}\033[0m\n" . str_repeat('═', 80) . "\n"; }
    public static function section($msg) { echo "\n\033[1;36m{$msg}\033[0m\n" . str_repeat('─', 80) . "\n"; }
}

$passed = 0;
$failed = 0;
$errors = [];

echo "\033[1m\n╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║       TESTE E2E - REQUISIÇÕES HTTP REAIS (Simulação de Navegador)           ║\n";
echo "║  Encontra erros que só aparecem em produção (Collection, View, etc)         ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\033[0m\n";

/**
 * Helper para simular requisição HTTP completa
 */
function simulateHttpRequest($method, $uri, $params = [], $headers = []) {
    global $kernel;
    
    try {
        $request = Request::create($uri, $method, $params, [], [], $headers);
        
        // Processar a requisição através do kernel
        $response = $kernel->handle($request);
        
        return [
            'success' => true,
            'status' => $response->getStatusCode(),
            'content' => $response->getContent(),
            'response' => $response,
            'exception' => null
        ];
        
    } catch (\Exception $e) {
        return [
            'success' => false,
            'status' => 500,
            'content' => null,
            'response' => null,
            'exception' => $e
        ];
    }
}

/**
 * Helper para criar usuário autenticado
 */
function authenticatedUser() {
    static $user = null;
    
    if ($user === null) {
        // Buscar primeiro usuário aprovado
        $user = User::where('status_cadastro', 'aprovado')->first();
        
        if (!$user) {
            // Criar usuário admin para testes
            $user = User::create([
                'name' => 'Admin Test',
                'email' => 'admin.test@test.com',
                'password' => bcrypt('password'),
                'status_cadastro' => 'aprovado'
            ]);
        }
    }
    
    return $user;
}

// ══════════════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 1: TESTES DE ROTAS PÚBLICAS (Guest)');

TestOutput::section('1.1 - GET / (Página Inicial)');

$result = simulateHttpRequest('GET', '/');

if ($result['success']) {
    TestOutput::success("GET / executado com sucesso");
    TestOutput::info("  → Status: {$result['status']}");
    $passed++;
    
    if ($result['status'] === 200) {
        TestOutput::success("Status 200 OK");
        $passed++;
    } else {
        TestOutput::warning("Status diferente de 200: {$result['status']}");
    }
} else {
    TestOutput::error("GET / falhou!");
    TestOutput::info("  → Erro: " . $result['exception']->getMessage());
    $errors[] = "GET / - " . $result['exception']->getMessage();
    $failed++;
}

TestOutput::section('1.2 - GET /register (Formulário de Registro)');

$result = simulateHttpRequest('GET', '/register');

if ($result['success']) {
    TestOutput::success("GET /register executado com sucesso");
    TestOutput::info("  → Status: {$result['status']}");
    $passed++;
    
    if ($result['status'] === 200) {
        TestOutput::success("Status 200 OK");
        $passed++;
        
        // Verificar se contém elementos esperados
        if (strpos($result['content'], 'codigo_convite') !== false) {
            TestOutput::success("Formulário contém campo 'codigo_convite'");
            $passed++;
        } else {
            TestOutput::warning("Campo 'codigo_convite' não encontrado no HTML");
        }
    }
} else {
    TestOutput::error("GET /register falhou!");
    TestOutput::info("  → Erro: " . $result['exception']->getMessage());
    $errors[] = "GET /register - " . $result['exception']->getMessage();
    $failed++;
}

TestOutput::section('1.3 - POST /register/validate-code (Validação AJAX)');

$franquia = Empresa::where('tipo_empresa', 'franquia')
    ->where('status', 0)
    ->whereNotNull('codigo_convite')
    ->first();

if ($franquia) {
    $result = simulateHttpRequest('POST', '/register/validate-code', [
        'codigo' => $franquia->codigo_convite
    ], [
        'HTTP_X_REQUESTED_WITH' => 'XMLHttpRequest',
        'HTTP_ACCEPT' => 'application/json'
    ]);
    
    if ($result['success']) {
        TestOutput::success("POST /register/validate-code executado");
        TestOutput::info("  → Status: {$result['status']}");
        $passed++;
        
        $json = json_decode($result['content'], true);
        
        if ($json && isset($json['valid']) && $json['valid'] === true) {
            TestOutput::success("Validação retornou JSON correto");
            TestOutput::info("  → Empresa: " . ($json['empresa']['name'] ?? 'N/A'));
            $passed++;
        } else {
            TestOutput::error("JSON inválido ou código não validado");
            $failed++;
        }
    } else {
        TestOutput::error("POST /register/validate-code falhou!");
        TestOutput::info("  → Erro: " . $result['exception']->getMessage());
        $errors[] = "POST /register/validate-code - " . $result['exception']->getMessage();
        $failed++;
    }
} else {
    TestOutput::warning("Nenhuma franquia disponível para teste");
}

// ══════════════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 2: TESTES DE ROTAS AUTENTICADAS (Principais Controllers)');

TestOutput::section('2.1 - GET /users (Lista de Usuários) - TESTE DO BUG REPORTADO');

// Autenticar usuário para simular sessão
$user = authenticatedUser();
auth()->login($user);

$result = simulateHttpRequest('GET', '/users');

if ($result['success']) {
    TestOutput::success("GET /users executado com sucesso");
    TestOutput::info("  → Status: {$result['status']}");
    $passed++;
    
    if ($result['status'] === 200) {
        TestOutput::success("Status 200 OK");
        $passed++;
        
        // Verificar se não há erros no HTML
        if (strpos($result['content'], 'currentPage does not exist') !== false) {
            TestOutput::error("🐛 BUG ENCONTRADO: Erro 'currentPage does not exist' na view!");
            TestOutput::info("  → UserController retorna Collection em vez de Paginator");
            $errors[] = "GET /users - currentPage() chamado em Collection (Controller não usa paginate())";
            $failed++;
        } elseif (strpos($result['content'], 'Call to undefined method') !== false) {
            TestOutput::error("🐛 BUG ENCONTRADO: Método indefinido na view!");
            $errors[] = "GET /users - Método indefinido chamado na view";
            $failed++;
        } else {
            TestOutput::success("Nenhum erro detectado no HTML");
            $passed++;
        }
    } else {
        TestOutput::warning("Status diferente de 200: {$result['status']}");
    }
} else {
    TestOutput::error("GET /users falhou!");
    TestOutput::info("  → Erro: " . $result['exception']->getMessage());
    TestOutput::info("  → Arquivo: " . $result['exception']->getFile() . ":" . $result['exception']->getLine());
    $errors[] = "GET /users - " . $result['exception']->getMessage();
    $failed++;
}

TestOutput::section('2.2 - GET /users/create (Formulário de Criação)');

$result = simulateHttpRequest('GET', '/users/create');

if ($result['success']) {
    TestOutput::success("GET /users/create executado");
    TestOutput::info("  → Status: {$result['status']}");
    $passed++;
    
    if ($result['status'] === 200) {
        TestOutput::success("Status 200 OK");
        $passed++;
    }
} else {
    TestOutput::error("GET /users/create falhou!");
    TestOutput::info("  → Erro: " . $result['exception']->getMessage());
    $errors[] = "GET /users/create - " . $result['exception']->getMessage();
    $failed++;
}

TestOutput::section('2.3 - GET /users/{id}/edit (Formulário de Edição)');

$testUser = User::first();

if ($testUser) {
    $result = simulateHttpRequest('GET', "/users/{$testUser->id}/edit");
    
    if ($result['success']) {
        TestOutput::success("GET /users/{$testUser->id}/edit executado");
        TestOutput::info("  → Status: {$result['status']}");
        $passed++;
        
        if ($result['status'] === 200) {
            TestOutput::success("Status 200 OK");
            $passed++;
        }
    } else {
        TestOutput::error("GET /users/{$testUser->id}/edit falhou!");
        TestOutput::info("  → Erro: " . $result['exception']->getMessage());
        $errors[] = "GET /users/edit - " . $result['exception']->getMessage();
        $failed++;
    }
} else {
    TestOutput::warning("Nenhum usuário disponível para teste de edição");
}

// ══════════════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 3: TESTES DE OUTROS RECURSOS (CRUD)');

TestOutput::section('3.1 - GET /empresas (Lista de Empresas)');

$result = simulateHttpRequest('GET', '/empresas');

if ($result['success']) {
    TestOutput::success("GET /empresas executado");
    TestOutput::info("  → Status: {$result['status']}");
    $passed++;
    
    if ($result['status'] === 200) {
        TestOutput::success("Status 200 OK");
        $passed++;
        
        // Verificar se há erros de Collection vs Paginator
        if (strpos($result['content'], 'currentPage does not exist') !== false ||
            strpos($result['content'], 'perPage does not exist') !== false) {
            TestOutput::error("🐛 BUG: Erro de paginação em /empresas");
            $errors[] = "GET /empresas - Erro de Collection/Paginator";
            $failed++;
        } else {
            TestOutput::success("Sem erros de paginação");
            $passed++;
        }
    }
} else {
    TestOutput::error("GET /empresas falhou!");
    TestOutput::info("  → Erro: " . $result['exception']->getMessage());
    $errors[] = "GET /empresas - " . $result['exception']->getMessage();
    $failed++;
}

TestOutput::section('3.2 - GET /groups (Lista de Grupos)');

$result = simulateHttpRequest('GET', '/groups');

if ($result['success']) {
    TestOutput::success("GET /groups executado");
    TestOutput::info("  → Status: {$result['status']}");
    $passed++;
    
    if ($result['status'] === 200) {
        TestOutput::success("Status 200 OK");
        $passed++;
        
        if (strpos($result['content'], 'currentPage does not exist') !== false ||
            strpos($result['content'], 'perPage does not exist') !== false) {
            TestOutput::error("🐛 BUG: Erro de paginação em /groups");
            $errors[] = "GET /groups - Erro de Collection/Paginator";
            $failed++;
        } else {
            TestOutput::success("Sem erros de paginação");
            $passed++;
        }
    }
} else {
    TestOutput::error("GET /groups falhou!");
    TestOutput::info("  → Erro: " . $result['exception']->getMessage());
    $errors[] = "GET /groups - " . $result['exception']->getMessage();
    $failed++;
}

TestOutput::section('3.3 - GET /permissions (Lista de Permissões)');

$result = simulateHttpRequest('GET', '/permissions');

if ($result['success']) {
    TestOutput::success("GET /permissions executado");
    TestOutput::info("  → Status: {$result['status']}");
    $passed++;
    
    if ($result['status'] === 200) {
        TestOutput::success("Status 200 OK");
        $passed++;
        
        if (strpos($result['content'], 'currentPage does not exist') !== false ||
            strpos($result['content'], 'perPage does not exist') !== false) {
            TestOutput::error("🐛 BUG: Erro de paginação em /permissions");
            $errors[] = "GET /permissions - Erro de Collection/Paginator";
            $failed++;
        } else {
            TestOutput::success("Sem erros de paginação");
            $passed++;
        }
    }
} else {
    TestOutput::error("GET /permissions falhou!");
    TestOutput::info("  → Erro: " . $result['exception']->getMessage());
    $errors[] = "GET /permissions - " . $result['exception']->getMessage();
    $failed++;
}

TestOutput::section('3.4 - GET /midias (Lista de Mídias)');

$result = simulateHttpRequest('GET', '/midias');

if ($result['success']) {
    TestOutput::success("GET /midias executado");
    TestOutput::info("  → Status: {$result['status']}");
    $passed++;
    
    if ($result['status'] === 200) {
        TestOutput::success("Status 200 OK");
        $passed++;
        
        if (strpos($result['content'], 'currentPage does not exist') !== false ||
            strpos($result['content'], 'perPage does not exist') !== false) {
            TestOutput::error("🐛 BUG: Erro de paginação em /midias");
            $errors[] = "GET /midias - Erro de Collection/Paginator";
            $failed++;
        } else {
            TestOutput::success("Sem erros de paginação");
            $passed++;
        }
    }
} else {
    TestOutput::error("GET /midias falhou!");
    TestOutput::info("  → Erro: " . $result['exception']->getMessage());
    $errors[] = "GET /midias - " . $result['exception']->getMessage();
    $failed++;
}

// ══════════════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 4: ANÁLISE DE ERROS COMUNS');

TestOutput::section('4.1 - Verificar Controllers que usam ::all() em vez de ::paginate()');

$controllersToCheck = [
    'UserController' => '/users',
    'EmpresaController' => '/empresas',
    'GroupController' => '/groups',
    'PermissionController' => '/permissions',
    'MidiaController' => '/midias',
];

foreach ($controllersToCheck as $controller => $route) {
    $controllerPath = app_path("Http/Controllers/{$controller}.php");
    
    if (file_exists($controllerPath)) {
        $content = file_get_contents($controllerPath);
        
        // Verificar se usa ::all() no método index
        if (preg_match('/function\s+index\s*\([^)]*\)\s*\{[^}]*::all\(\)/s', $content)) {
            TestOutput::error("🐛 {$controller} usa ::all() no index (deveria usar ::paginate())");
            $errors[] = "{$controller} - Usa ::all() em vez de ::paginate()";
            $failed++;
        } else {
            TestOutput::success("{$controller} não usa ::all() no index");
            $passed++;
        }
    }
}

// ══════════════════════════════════════════════════════════════════════════════
TestOutput::title('FASE 5: LIMPEZA');

// Desautenticar
auth()->logout();

// Remover usuário de teste se foi criado
$testUser = User::where('email', 'admin.test@test.com')->first();
if ($testUser && $testUser->created_at > now()->subMinutes(5)) {
    $testUser->forceDelete();
    TestOutput::success("Usuário de teste removido");
    $passed++;
} else {
    TestOutput::info("Usuário de teste mantido (é admin real do sistema)");
    $passed++;
}

// ══════════════════════════════════════════════════════════════════════════════
echo "\n\033[1m╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                    RESUMO - TESTE HTTP E2E                                   ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\033[0m\n\n";

echo "Total de testes executados: \033[1m" . ($passed + $failed) . "\033[0m\n";
echo "\033[32m✓ Testes passados: {$passed}\033[0m\n";

if ($failed > 0) {
    echo "\033[31m✗ Testes falhados: {$failed}\033[0m\n";
}

$percentage = ($passed + $failed) > 0 ? round(($passed / ($passed + $failed)) * 100, 2) : 0;

echo "\n\033[1mTaxa de sucesso: ";
if ($percentage === 100.0) {
    echo "\033[32m{$percentage}%\033[0m ✓\n";
} elseif ($percentage >= 80) {
    echo "\033[33m{$percentage}%\033[0m ⚠️\n";
} else {
    echo "\033[31m{$percentage}%\033[0m ✗\n";
}

if (!empty($errors)) {
    echo "\n\033[1;31m╔══════════════════════════════════════════════════════════════════════════════╗\n";
    echo "║                           🐛 BUGS ENCONTRADOS                                ║\n";
    echo "╚══════════════════════════════════════════════════════════════════════════════╝\033[0m\n\n";
    
    foreach ($errors as $i => $error) {
        echo "\033[31m" . ($i + 1) . ". {$error}\033[0m\n";
    }
}

echo "\n\033[36m════════════════════════════════════════════════════════════════════════════════\033[0m\n";
echo "\033[1mCOBERTURA DO TESTE HTTP:\033[0m\n";
echo "  ✓ Rotas públicas (GET /, /register, POST /register/validate-code)\n";
echo "  ✓ Rotas autenticadas (CRUD de Users, Empresas, Groups, Permissions, Midias)\n";
echo "  ✓ Detecção de erros Collection vs Paginator\n";
echo "  ✓ Análise estática de código (::all() vs ::paginate())\n";
echo "  ✓ Validação de status HTTP\n";
echo "  ✓ Validação de conteúdo HTML/JSON\n";
echo "\033[36m════════════════════════════════════════════════════════════════════════════════\033[0m\n\n";

exit($failed > 0 ? 1 : 0);
