#!/usr/bin/env php
<?php

/**
 * Teste de Feature Flags - Sistema de Emails
 * ===========================================
 * Verifica configuração e funcionamento das feature flags
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use Illuminate\Support\Facades\Config;

echo "\n";
echo "╔═══════════════════════════════════════════════════════════╗\n";
echo "║    TESTE DE FEATURE FLAGS - Sistema de Emails            ║\n";
echo "╚═══════════════════════════════════════════════════════════╝\n";
echo "\n";

// Ler configurações atuais
echo "📊 CONFIGURAÇÕES ATUAIS:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$emailEnabled = Config::get('features.email_enabled', true);
$devMode = Config::get('features.dev_mode', false);
$testRecipient = Config::get('features.test_recipient');

// Flag Mestre
$icon = $emailEnabled ? '✅' : '❌';
echo "  {$icon} FEATURE_EMAIL_ENABLED: " . ($emailEnabled ? 'true' : 'false') . "\n";
if (!$emailEnabled) {
    echo "     ⚠️  Todos os emails estão DESABILITADOS\n";
}
echo "\n";

// Flags Individuais
$tipos = [
    'cadastro_aprovado' => 'Cadastro Aprovado (verde ✅)',
    'cadastro_rejeitado' => 'Cadastro Rejeitado (vermelho ❌)',
    'novo_cadastro_pendente' => 'Novo Cadastro Pendente (laranja 🔔)',
];

echo "  📧 EMAILS INDIVIDUAIS:\n";
foreach ($tipos as $key => $descricao) {
    $enabled = Config::get("features.emails.{$key}", true);
    $icon = $enabled ? '✅' : '❌';
    $status = $enabled ? 'HABILITADO' : 'DESABILITADO';
    echo "     {$icon} {$descricao}: {$status}\n";
}
echo "\n";

// Modo Desenvolvimento
$icon = $devMode ? '🔧' : '🚀';
echo "  {$icon} FEATURE_EMAIL_DEV_MODE: " . ($devMode ? 'true' : 'false') . "\n";
if ($devMode) {
    echo "     📝 Emails serão apenas LOGADOS (não enviados)\n";
} else {
    echo "     📬 Emails serão ENVIADOS via SMTP\n";
}
echo "\n";

// Email de Teste
if (!empty($testRecipient)) {
    echo "  🧪 FEATURE_EMAIL_TEST_RECIPIENT: {$testRecipient}\n";
    echo "     ⚠️  Todos os emails serão REDIRECIONADOS para este endereço\n";
} else {
    echo "  ✅ FEATURE_EMAIL_TEST_RECIPIENT: (não configurado)\n";
    echo "     📨 Emails serão enviados aos destinatários reais\n";
}
echo "\n";

// Análise do Comportamento
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "🔍 ANÁLISE DO COMPORTAMENTO:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

if (!$emailEnabled) {
    echo "  ❌ TODOS OS EMAILS DESABILITADOS\n";
    echo "     • Nenhum email será enviado ou logado\n";
    echo "     • Sistema continua funcionando normalmente\n";
    echo "     • Usuários verão: '(envio de email desabilitado)'\n\n";
} else {
    if ($devMode) {
        echo "  🔧 MODO DESENVOLVIMENTO ATIVO\n";
        echo "     • Emails serão LOGADOS em storage/logs/laravel.log\n";
        echo "     • Nenhum email será enviado de verdade\n";
        echo "     • Não consome quota de SMTP\n";
        echo "     • Ideal para desenvolvimento/debug\n\n";
        
        echo "  📝 Ver logs:\n";
        echo "     tail -f storage/logs/laravel.log | grep 'dev mode'\n\n";
    } else {
        echo "  🚀 MODO PRODUÇÃO ATIVO\n";
        echo "     • Emails serão ENVIADOS via SMTP\n";
        echo "     • Servidor: " . Config::get('mail.mailers.smtp.host') . "\n";
        echo "     • Porta: " . Config::get('mail.mailers.smtp.port') . "\n";
        echo "     • Remetente: " . Config::get('mail.from.address') . "\n\n";
        
        if (!empty($testRecipient)) {
            echo "  ⚠️  REDIRECIONAMENTO ATIVO:\n";
            echo "     • Emails irão para: {$testRecipient}\n";
            echo "     • Destinatários originais: IGNORADOS\n";
            echo "     • Útil para testes/homologação\n\n";
        }
    }
    
    // Verificar quais emails estão ativos
    $ativos = [];
    $inativos = [];
    
    foreach ($tipos as $key => $descricao) {
        if (Config::get("features.emails.{$key}", true)) {
            $ativos[] = $descricao;
        } else {
            $inativos[] = $descricao;
        }
    }
    
    if (!empty($ativos)) {
        echo "  ✅ EMAILS ATIVOS:\n";
        foreach ($ativos as $desc) {
            echo "     • {$desc}\n";
        }
        echo "\n";
    }
    
    if (!empty($inativos)) {
        echo "  ❌ EMAILS DESABILITADOS:\n";
        foreach ($inativos as $desc) {
            echo "     • {$desc}\n";
        }
        echo "\n";
    }
}

// Recomendações
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "💡 RECOMENDAÇÕES:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

// Detectar ambiente
$env = Config::get('app.env');

if ($env === 'production') {
    echo "  🏭 AMBIENTE: PRODUÇÃO\n\n";
    
    if ($devMode) {
        echo "  ⚠️  ATENÇÃO: Dev mode ativo em produção!\n";
        echo "     • Emails não estão sendo enviados\n";
        echo "     • Defina: FEATURE_EMAIL_DEV_MODE=false\n\n";
    }
    
    if (!empty($testRecipient)) {
        echo "  ⚠️  ATENÇÃO: Email de teste configurado em produção!\n";
        echo "     • Emails sendo redirecionados\n";
        echo "     • Defina: FEATURE_EMAIL_TEST_RECIPIENT=null\n\n";
    }
    
    if (!$emailEnabled) {
        echo "  ⚠️  ATENÇÃO: Emails desabilitados em produção!\n";
        echo "     • Usuários não receberão notificações\n";
        echo "     • Defina: FEATURE_EMAIL_ENABLED=true\n\n";
    }
    
    if ($emailEnabled && !$devMode && empty($testRecipient)) {
        echo "  ✅ Configuração ideal para produção!\n\n";
    }
    
} elseif ($env === 'local') {
    echo "  💻 AMBIENTE: DESENVOLVIMENTO\n\n";
    
    if (!$devMode && empty($testRecipient)) {
        echo "  💡 Sugestão: Ativar modo desenvolvimento\n";
        echo "     • Defina: FEATURE_EMAIL_DEV_MODE=true\n";
        echo "     • Emails serão apenas logados\n";
        echo "     • Não consome SMTP\n\n";
    }
    
    if ($devMode) {
        echo "  ✅ Dev mode ativo - emails sendo logados!\n\n";
    }
    
} else {
    echo "  🧪 AMBIENTE: {$env}\n\n";
    
    if (!empty($testRecipient)) {
        echo "  ✅ Email de teste configurado - bom para homologação!\n\n";
    }
}

// Comandos Úteis
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "🛠️  COMANDOS ÚTEIS:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "  # Desabilitar todos os emails\n";
echo "  FEATURE_EMAIL_ENABLED=false\n\n";

echo "  # Ativar modo desenvolvimento\n";
echo "  FEATURE_EMAIL_DEV_MODE=true\n\n";

echo "  # Redirecionar para teste\n";
echo "  FEATURE_EMAIL_TEST_RECIPIENT=seu_email@example.com\n\n";

echo "  # Desabilitar apenas um tipo\n";
echo "  FEATURE_EMAIL_NOVO_CADASTRO_PENDENTE=false\n\n";

echo "  # Limpar cache após alterar .env\n";
echo "  php artisan config:clear\n\n";

echo "  # Ver logs em tempo real\n";
echo "  tail -f storage/logs/laravel.log\n\n";

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "📖 Documentação completa: FEATURE-FLAGS-EMAILS.md\n\n";
