<?php

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use App\Models\Empresa;
use App\Models\Midia;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;
use Illuminate\Http\UploadedFile;

echo "\n";
echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║                                                                ║\n";
echo "║   🧪 TESTE DE UPLOAD REAL COM IMAGEM EXISTENTE                ║\n";
echo "║                                                                ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

// 1. Buscar empresa
echo "1️⃣  BUSCANDO EMPRESA...\n";
$empresa = Empresa::first();
if (!$empresa) {
    echo "❌ Nenhuma empresa encontrada!\n";
    exit(1);
}
echo "   ✅ Empresa: {$empresa->name}\n";
echo "   🔑 Código: {$empresa->codigo_convite}\n\n";

// 2. Selecionar imagem de exemplo
echo "2️⃣  SELECIONANDO IMAGEM DE EXEMPLO...\n";
$sourceImage = '/tmp/teste_real.jpg';

if (!file_exists($sourceImage)) {
    echo "❌ Imagem não encontrada: {$sourceImage}\n";
    exit(1);
}

$imageInfo = getimagesize($sourceImage);
$sourceSize = filesize($sourceImage);

echo "   📂 Arquivo: " . basename($sourceImage) . "\n";
echo "   📐 Dimensões originais: {$imageInfo[0]}x{$imageInfo[1]}\n";
echo "   💾 Tamanho: " . round($sourceSize / 1024, 2) . " KB\n\n";

// 3. Processar upload (simulando o controller)
echo "3️⃣  PROCESSANDO UPLOAD (COMO CONTROLLER)...\n";

$fileContent = file_get_contents($sourceImage);
$extension = 'jpg';

// Gerar nome criptografado MD5 (mesmo código do controller)
$uniqueString = uniqid('', true) . microtime(true) . rand(1000, 9999);
$hashedName = md5($uniqueString);
$filename = $hashedName . '.' . $extension;

echo "   🔐 Hash MD5: {$hashedName}\n";
echo "   📝 Filename: {$filename}\n\n";

// 4. Redimensionar com Intervention Image
echo "4️⃣  REDIMENSIONANDO PARA 640x1100...\n";
$manager = new ImageManager(new Driver());
$image = $manager->read($fileContent);

echo "   📐 Original: {$image->width()}x{$image->height()}\n";
$image->resize(640, 1100);
echo "   📐 Redimensionada: 640x1100\n\n";

// 5. Salvar no storage
echo "5️⃣  SALVANDO NO STORAGE...\n";
$path = 'midias/' . $filename;
Storage::disk('public')->put($path, (string) $image->encode());

$fullPath = storage_path('app/public/' . $path);
$savedSize = file_exists($fullPath) ? filesize($fullPath) : 0;

echo "   ✅ Path: {$path}\n";
echo "   📂 Full path: {$fullPath}\n";
echo "   💾 Tamanho salvo: " . round($savedSize / 1024, 2) . " KB\n\n";

// 6. Criar registro no banco
echo "6️⃣  CRIANDO REGISTRO NO BANCO...\n";
$midia = Midia::create([
    'title' => 'Upload Real - Imagem de Exemplo - ' . date('Y-m-d H:i:s'),
    'tipo' => 'imagem',
    'formato' => 'jpg',
    'path' => $path,
    'empresa_legado_id' => $empresa->id,
    'position' => 1,
    'start_datetime' => now(),
    'end_datetime' => now()->addDays(30),
    'status_aprovacao' => 'aprovado',
]);

echo "   ✅ ID: {$midia->id}\n";
echo "   📝 Título: {$midia->title}\n";
echo "   📂 Path: {$midia->path}\n\n";

// 7. Gerar URLs
echo "7️⃣  TESTANDO URLs...\n";
$storageUrl = Storage::url($midia->path);
$appUrl = config('app.url');
$fullUrl = $appUrl . $storageUrl;

echo "   🔗 Storage::url(): {$storageUrl}\n";
echo "   🔗 APP_URL: {$appUrl}\n";
echo "   🔗 URL completa: {$fullUrl}\n";
echo "   🔗 URL relativa: /storage/{$path}\n\n";

// 8. Validações
echo "8️⃣  VALIDAÇÕES FINAIS...\n";

$publicPath = public_path('storage/' . $path);
$validations = [
    'Arquivo em storage/app/public/' => file_exists($fullPath),
    'Arquivo em public/storage/ (via link)' => file_exists($publicPath),
    'Campo path no banco' => !empty($midia->path),
    'Tamanho > 0' => $savedSize > 0,
    'Storage::exists()' => Storage::disk('public')->exists($path),
    'Formato correto' => $midia->formato === 'jpg',
    'Tipo correto' => $midia->tipo === 'imagem',
];

$allPassed = true;
foreach ($validations as $test => $passed) {
    $status = $passed ? '✅' : '❌';
    echo "   {$status} {$test}\n";
    if (!$passed) $allPassed = false;
}

echo "\n";

// 9. Testar acesso HTTP
echo "9️⃣  TESTANDO ACESSO HTTP...\n";
$testUrl = "http://localhost/storage/{$path}";
echo "   🔗 URL de teste: {$testUrl}\n";

// Usar curl interno para testar
$ch = curl_init($testUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HEADER, true);
curl_setopt($ch, CURLOPT_NOBODY, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 5);
$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpCode == 200) {
    echo "   ✅ HTTP {$httpCode} - Imagem acessível!\n";
} else {
    echo "   ❌ HTTP {$httpCode} - Erro ao acessar imagem\n";
    if ($httpCode == 403) {
        echo "   ⚠️  Erro 403: Verificar permissões ou configuração do servidor\n";
    }
}

echo "\n";
echo "╔════════════════════════════════════════════════════════════════╗\n";
echo "║                                                                ║\n";
if ($allPassed && $httpCode == 200) {
    echo "║   🎉 UPLOAD COMPLETO E IMAGEM ACESSÍVEL! 🎉                   ║\n";
} elseif ($allPassed) {
    echo "║   ⚠️  UPLOAD OK, MAS IMAGEM COM PROBLEMA DE ACESSO HTTP       ║\n";
} else {
    echo "║   ❌ ALGUNS TESTES FALHARAM                                   ║\n";
}
echo "║                                                                ║\n";
echo "╚════════════════════════════════════════════════════════════════╝\n\n";

echo "📊 RESUMO FINAL:\n";
echo "   • ID: {$midia->id}\n";
echo "   • Filename: {$filename}\n";
echo "   • Path: {$midia->path}\n";
echo "   • URL Storage: {$storageUrl}\n";
echo "   • URL HTTP: {$testUrl}\n";
echo "   • Status HTTP: {$httpCode}\n\n";

// 10. Instruções para visualizar
echo "🔍 PARA VISUALIZAR A IMAGEM:\n";
echo "   1. Acesse: http://localhost:8082/midias\n";
echo "   2. Procure pelo criativo ID {$midia->id}\n";
echo "   3. Clique no ícone de preview (imagem)\n";
echo "   4. Ou acesse direto: http://localhost:8082/storage/{$path}\n\n";

exit($allPassed ? 0 : 1);
