# 🔌 API DE CONSULTA DE PUBLICAÇÕES - LÓGICA DE PRIORIDADE

## 📋 Visão Geral

Quando o **equipamento frontend** solicita publicações, ele deve seguir uma **ordem de prioridade** para determinar quais publicações exibir.

---

## 🎯 LÓGICA DE PRIORIDADE

### **REGRA: Serial Específico TEM PRIORIDADE sobre Empresa**

```
1º) Buscar publicações para o SERIAL específico
2º) Se NÃO encontrar, buscar publicações para a EMPRESA
3º) SEMPRE incluir publicações para TODOS
```

---

## 📡 ENDPOINT DA API (Exemplo)

### **GET /api/publicacoes/equipamento**

**Parâmetros:**
```json
{
  "serial": "ABC123456",
  "empresa_id": 5
}
```

**Headers:**
```
Authorization: Bearer {token}
Content-Type: application/json
```

---

## 🔍 QUERY SQL OTIMIZADA

### **Versão 1: COM prioridade de Serial**

```sql
-- PASSO 1: Buscar publicações para o serial específico
SELECT * FROM publicacoes
WHERE tipo_alcance = 'serial_especifico'
  AND equipment_serial = 'ABC123456'
  AND status = 'ativa'
  AND data_inicio <= NOW()
  AND data_fim >= NOW()
LIMIT 1;

-- SE encontrou publicação para serial, PARAR AQUI e não buscar empresa
-- SE NÃO encontrou, continuar...

-- PASSO 2: Buscar publicações para a empresa (SOMENTE SE não achou serial)
SELECT * FROM publicacoes
WHERE tipo_alcance = 'empresa'
  AND empresa_alcance_id = 5
  AND status = 'ativa'
  AND data_inicio <= NOW()
  AND data_fim >= NOW();

-- PASSO 3: Buscar publicações para TODOS (sempre incluir)
SELECT * FROM publicacoes
WHERE tipo_alcance = 'todos'
  AND status = 'ativa'
  AND data_inicio <= NOW()
  AND data_fim >= NOW();

-- RESULTADO FINAL: UNION dos passos 1 OU 2 + passo 3
```

---

## 💻 IMPLEMENTAÇÃO EM LARAVEL (Controller)

### **Arquivo: app/Http/Controllers/Api/PublicacaoApiController.php**

```php
<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Publicacao;
use Illuminate\Http\Request;

class PublicacaoApiController extends Controller
{
    /**
     * Buscar publicações para um equipamento específico
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function paraEquipamento(Request $request)
    {
        $validated = $request->validate([
            'serial' => 'required|string|max:255',
            'empresa_id' => 'required|integer|exists:empresas,id',
        ]);

        $serial = $validated['serial'];
        $empresaId = $validated['empresa_id'];

        // Base query: publicações ativas e vigentes
        $baseQuery = Publicacao::with(['criativo', 'franquia', 'cliente'])
            ->ativas()
            ->vigentes();

        // PASSO 1: Buscar publicações para o serial específico
        $publicacoesSerial = (clone $baseQuery)
            ->serialEspecifico($serial)
            ->get();

        // PASSO 2: Buscar publicações para a empresa
        // SOMENTE SE não encontrou publicações para o serial
        $publicacoesEmpresa = collect();
        if ($publicacoesSerial->isEmpty()) {
            $publicacoesEmpresa = (clone $baseQuery)
                ->paraEmpresa($empresaId)
                ->get();
        }

        // PASSO 3: Buscar publicações para TODOS (sempre incluir)
        $publicacoesTodos = (clone $baseQuery)
            ->paraTodos()
            ->get();

        // MERGE: Serial OU Empresa + Todos
        $publicacoes = $publicacoesSerial
            ->merge($publicacoesEmpresa)
            ->merge($publicacoesTodos)
            ->unique('id')  // Remover duplicatas
            ->values();     // Reindexar

        return response()->json([
            'success' => true,
            'total' => $publicacoes->count(),
            'equipamento' => [
                'serial' => $serial,
                'empresa_id' => $empresaId,
            ],
            'prioridade' => [
                'serial_especifico' => $publicacoesSerial->count(),
                'empresa' => $publicacoesEmpresa->count(),
                'todos' => $publicacoesTodos->count(),
            ],
            'data' => $publicacoes,
        ]);
    }
}
```

---

## 🧪 EXEMPLOS DE CENÁRIOS

### **Cenário 1: Equipamento com publicação específica**

**Equipamento:**
- Serial: ABC123456
- Empresa: Shopping Iguatemi (ID 5)

**Banco de Dados:**
```
Pub #1: tipo=todos (Black Friday)
Pub #2: tipo=serial_especifico, serial=ABC123456 (Teste A/B)
Pub #3: tipo=empresa, empresa_id=5 (Promo Iguatemi)
```

**Resultado:**
```json
{
  "total": 2,
  "prioridade": {
    "serial_especifico": 1,  // ✅ Encontrou
    "empresa": 0,             // ❌ Ignorado (serial tem prioridade)
    "todos": 1
  },
  "data": [
    { "id": 2, "titulo": "Teste A/B", "tipo_alcance": "serial_especifico" },
    { "id": 1, "titulo": "Black Friday", "tipo_alcance": "todos" }
  ]
}
```

**📝 Nota:** Pub #3 (empresa) **NÃO** foi incluída porque Pub #2 (serial) tem prioridade!

---

### **Cenário 2: Equipamento SEM publicação específica**

**Equipamento:**
- Serial: XYZ999 (sem publicação específica)
- Empresa: Shopping Iguatemi (ID 5)

**Banco de Dados:**
```
Pub #1: tipo=todos (Black Friday)
Pub #2: tipo=serial_especifico, serial=ABC123456 (Teste A/B - outro serial)
Pub #3: tipo=empresa, empresa_id=5 (Promo Iguatemi)
```

**Resultado:**
```json
{
  "total": 2,
  "prioridade": {
    "serial_especifico": 0,  // ❌ Não encontrou
    "empresa": 1,             // ✅ Incluído (fallback)
    "todos": 1
  },
  "data": [
    { "id": 3, "titulo": "Promo Iguatemi", "tipo_alcance": "empresa" },
    { "id": 1, "titulo": "Black Friday", "tipo_alcance": "todos" }
  ]
}
```

**📝 Nota:** Pub #3 (empresa) **FOI** incluída porque não há publicação específica para este serial!

---

### **Cenário 3: Equipamento sem nada específico**

**Equipamento:**
- Serial: DEF789
- Empresa: Loja ABC (ID 10 - sem publicações)

**Banco de Dados:**
```
Pub #1: tipo=todos (Black Friday)
Pub #2: tipo=serial_especifico, serial=ABC123456 (outro serial)
Pub #3: tipo=empresa, empresa_id=5 (outra empresa)
```

**Resultado:**
```json
{
  "total": 1,
  "prioridade": {
    "serial_especifico": 0,  // ❌ Não encontrou
    "empresa": 0,             // ❌ Não encontrou
    "todos": 1                // ✅ Sempre incluir
  },
  "data": [
    { "id": 1, "titulo": "Black Friday", "tipo_alcance": "todos" }
  ]
}
```

**📝 Nota:** Apenas publicações globais (todos) foram retornadas!

---

## 🚀 ROTA DA API

### **Arquivo: routes/api.php**

```php
use App\Http\Controllers\Api\PublicacaoApiController;

Route::middleware(['auth:sanctum'])->group(function () {
    // Buscar publicações para equipamento
    Route::get('/publicacoes/equipamento', [PublicacaoApiController::class, 'paraEquipamento']);
});
```

---

## 📊 DIAGRAMA DE FLUXO

```
┌─────────────────────────────────────┐
│  Equipamento solicita publicações  │
│  Serial: ABC123, Empresa ID: 5     │
└─────────────────────────────────────┘
                  │
                  ▼
┌─────────────────────────────────────┐
│  PASSO 1: Buscar por SERIAL         │
│  WHERE tipo_alcance =               │
│    'serial_especifico'              │
│    AND serial = 'ABC123'            │
└─────────────────────────────────────┘
                  │
          ┌───────┴────────┐
          │                │
    Encontrou?         Não encontrou?
          │                │
          ▼                ▼
    ┌──────────┐    ┌──────────────┐
    │ ✅ PARAR │    │ PASSO 2:     │
    │ (não     │    │ Buscar por   │
    │ buscar   │    │ EMPRESA      │
    │ empresa) │    │ WHERE        │
    └──────────┘    │ empresa_id=5 │
          │         └──────────────┘
          │               │
          └───────┬───────┘
                  ▼
┌─────────────────────────────────────┐
│  PASSO 3: Buscar TODOS              │
│  WHERE tipo_alcance = 'todos'       │
└─────────────────────────────────────┘
                  │
                  ▼
┌─────────────────────────────────────┐
│  MERGE: Serial OU Empresa + Todos   │
│  Remover duplicatas                 │
│  Ordenar por prioridade             │
└─────────────────────────────────────┘
                  │
                  ▼
┌─────────────────────────────────────┐
│  RETORNAR JSON com publicações      │
└─────────────────────────────────────┘
```

---

## ⚡ OTIMIZAÇÕES

### **1. Cache de Publicações**

```php
use Illuminate\Support\Facades\Cache;

$cacheKey = "publicacoes:{$serial}:{$empresaId}";

$publicacoes = Cache::remember($cacheKey, 300, function() use ($serial, $empresaId) {
    // Lógica de busca aqui
    return $this->buscarPublicacoes($serial, $empresaId);
});
```

**TTL:** 5 minutos (300 segundos)

### **2. Eager Loading**

```php
Publicacao::with(['criativo', 'franquia', 'cliente', 'empresaAlcance'])
    ->ativas()
    ->vigentes()
    ->get();
```

### **3. Índices no Banco de Dados**

```sql
-- Migration
CREATE INDEX idx_pub_alcance ON publicacoes(tipo_alcance, status, data_inicio, data_fim);
CREATE INDEX idx_pub_serial ON publicacoes(equipment_serial) WHERE tipo_alcance = 'serial_especifico';
CREATE INDEX idx_pub_empresa ON publicacoes(empresa_alcance_id) WHERE tipo_alcance = 'empresa';
```

---

## 🔒 SEGURANÇA

### **1. Autenticação**

```php
Route::middleware(['auth:sanctum'])->group(function () {
    Route::get('/publicacoes/equipamento', ...);
});
```

### **2. Rate Limiting**

```php
Route::middleware(['throttle:60,1'])->group(function () {
    // Máximo 60 requisições por minuto
    Route::get('/publicacoes/equipamento', ...);
});
```

### **3. Validação de Equipamento**

```php
// Verificar se equipamento existe e está ativo
$equipamento = Equipamento::where('serial', $serial)
    ->where('empresa_id', $empresaId)
    ->where('status', 'ativo')
    ->firstOrFail();
```

---

## 📝 RESUMO

| Prioridade | Tipo | Lógica |
|------------|------|--------|
| **1º** | Serial Específico | Se encontrar, **PARAR** (não buscar empresa) |
| **2º** | Empresa | Buscar **SOMENTE SE** não encontrou serial |
| **3º** | Todos | **SEMPRE** incluir |

**Resultado Final:** `Serial OU Empresa` + `Todos`

---

## ✅ CHECKLIST DE IMPLEMENTAÇÃO

- [ ] Criar `PublicacaoApiController`
- [ ] Adicionar rota em `routes/api.php`
- [ ] Implementar lógica de prioridade
- [ ] Adicionar cache (5 minutos)
- [ ] Adicionar autenticação (Sanctum)
- [ ] Adicionar rate limiting (60/min)
- [ ] Criar índices no banco
- [ ] Testar cenários 1, 2 e 3
- [ ] Documentar endpoint na API
- [ ] Validar performance

---

**Criado em:** 18/10/2025  
**Sistema:** FreeFi Admin - API de Publicações  
**Versão:** 1.0
