# 🔍 Sistema de Filtro Avançado - Hotspots

## 📋 Visão Geral
Sistema de busca inteligente implementado no módulo de Hotspots, permitindo filtrar registros por diferentes campos de forma precisa e eficiente.

## ✨ Funcionalidades

### 1. Filtro por Tipo
Um select permite escolher o campo onde a busca será realizada:

- **Empresa**: Busca pelo nome da empresa associada
- **Marca**: Busca pela marca do hotspot
- **Modelo**: Busca pelo modelo do hotspot
- **Serial**: Busca pelo número de série

### 2. Busca Parcial (LIKE)
O sistema realiza busca parcial, ou seja:
- Digitando "Mikro" encontra "Mikrotik"
- Digitando "hAP" encontra "hAP ac2", "hAP ac3", etc.
- Não é case-sensitive (maiúsculas/minúsculas)

### 3. Interface Intuitiva
- **Placeholder dinâmico**: Muda conforme o tipo de filtro selecionado
- **Botão "Limpar"**: Aparece quando há filtros ativos
- **Preservação de filtros**: Mantém os filtros ao navegar entre páginas
- **Ícones visuais**: Indicadores claros para cada ação

## 🎯 Como Usar

### Passo a Passo:
1. Acesse a listagem de Hotspots
2. Selecione o **Tipo de Filtro** no select
3. Digite o **termo de busca** no campo de texto
4. Clique em **Buscar** (ou pressione Enter)
5. Os resultados serão filtrados automaticamente

### Exemplos Práticos:

#### Buscar por Empresa:
```
Tipo: Empresa
Busca: "Empresa A"
Resultado: Todos os hotspots da Empresa A
```

#### Buscar por Marca:
```
Tipo: Marca
Busca: "Mikrotik"
Resultado: Todos os hotspots Mikrotik
```

#### Buscar por Modelo:
```
Tipo: Modelo
Busca: "UniFi"
Resultado: Todos os modelos que contenham "UniFi"
```

#### Buscar por Serial:
```
Tipo: Serial
Busca: "AA1234"
Resultado: Hotspots com serial contendo "AA1234"
```

## 💻 Implementação Técnica

### Backend (Controller)
```php
public function index(Request $request)
{
    $query = Hotspot::with('empresa');

    if ($request->filled('filter_type') && $request->filled('search')) {
        $filterType = $request->input('filter_type');
        $searchTerm = $request->input('search');

        switch ($filterType) {
            case 'empresa':
                $query->whereHas('empresa', function($q) use ($searchTerm) {
                    $q->where('name', 'like', '%' . $searchTerm . '%');
                });
                break;
            case 'marca':
                $query->where('marca', 'like', '%' . $searchTerm . '%');
                break;
            case 'modelo':
                $query->where('modelo', 'like', '%' . $searchTerm . '%');
                break;
            case 'serial':
                $query->where('serial', 'like', '%' . $searchTerm . '%');
                break;
        }
    }

    $hotspots = $query->latest()->paginate(15)->withQueryString();
    return view('hotspots.index', compact('hotspots'));
}
```

### Frontend (View)
- **Select de Tipo**: 4 opções (empresa, marca, modelo, serial)
- **Input de Busca**: Campo de texto com placeholder dinâmico
- **Botão Buscar**: Submete o formulário
- **Botão Limpar**: Remove filtros ativos (aparece condicionalmente)

### JavaScript (UX Enhancement)
```javascript
// Placeholders dinâmicos por tipo
const placeholders = {
    'empresa': 'Digite o nome da empresa...',
    'marca': 'Digite a marca do hotspot...',
    'modelo': 'Digite o modelo do hotspot...',
    'serial': 'Digite o serial do hotspot...'
};

// Atualizar placeholder ao mudar o select
$('#filter_type').on('change', function() {
    updatePlaceholder();
    $('#search').focus();
});
```

## 🔧 Recursos Técnicos

### 1. Preservação de Estado
- `withQueryString()`: Mantém parâmetros na paginação
- `request('filter_type')`: Preserva tipo selecionado
- `request('search')`: Preserva termo de busca

### 2. Relacionamento (Busca por Empresa)
```php
$query->whereHas('empresa', function($q) use ($searchTerm) {
    $q->where('name', 'like', '%' . $searchTerm . '%');
});
```
Usa `whereHas` para buscar em relacionamento Eloquent

### 3. SQL Gerado
```sql
-- Busca por Marca
SELECT * FROM hotspots WHERE marca LIKE '%termo%'

-- Busca por Empresa
SELECT * FROM hotspots 
WHERE EXISTS (
    SELECT * FROM empresas 
    WHERE hotspots.empresa_id = empresas.id 
    AND empresas.name LIKE '%termo%'
)
```

## 📊 Performance

### Otimizações Implementadas:
- ✅ Eager Loading: `with('empresa')` carrega relações de uma vez
- ✅ Índices: `empresa_id` tem chave estrangeira indexada
- ✅ Paginação: Limite de 15 itens por página
- ✅ Busca condicional: Só executa WHERE quando há filtro

### Recomendações:
Para melhorar ainda mais a performance em grandes volumes:
- Adicionar índice em `marca`, `modelo`, `serial`
- Implementar cache de resultados frequentes
- Considerar full-text search para milhares de registros

## 🎨 Design Responsivo

### Desktop:
```
[Tipo: Empresa ▼] [Digite o nome da empresa...] [🔍 Buscar] [❌]
```

### Mobile:
Os elementos se reorganizam em stack vertical mantendo usabilidade

## 🚀 Expansão para Outros Módulos

Este sistema foi projetado para ser replicado facilmente:

### 1. Empresas
```php
// Filtros sugeridos: Nome, CNPJ, Email
case 'nome':
    $query->where('name', 'like', '%' . $searchTerm . '%');
    break;
case 'cnpj':
    $query->where('cnpj', 'like', '%' . $searchTerm . '%');
    break;
```

### 2. Mídias
```php
// Filtros sugeridos: Empresa, Nome, Tipo
case 'empresa':
    $query->whereHas('empresa', function($q) use ($searchTerm) {
        $q->where('name', 'like', '%' . $searchTerm . '%');
    });
    break;
```

### 3. Usuários
```php
// Filtros sugeridos: Nome, Email, Grupo
case 'grupo':
    $query->whereHas('groups', function($q) use ($searchTerm) {
        $q->where('name', 'like', '%' . $searchTerm . '%');
    });
    break;
```

## ✅ Checklist de Implementação em Novos Módulos

- [ ] Atualizar método `index()` do Controller
- [ ] Adicionar `withQueryString()` na paginação
- [ ] Criar formulário de busca na view
- [ ] Adicionar select com tipos de filtro
- [ ] Implementar JavaScript para placeholder dinâmico
- [ ] Adicionar botão "Limpar filtros"
- [ ] Testar todas as combinações de filtro
- [ ] Verificar preservação de estado na paginação
- [ ] Validar performance com dados reais
- [ ] Documentar novos tipos de filtro

## 📝 Notas de Desenvolvimento

### Data de Implementação: 17/10/2025
### Módulo Piloto: Hotspots
### Status: ✅ Implementado e Testado

### Próximos Passos:
1. ✅ Implementação no módulo Hotspots
2. ⏳ Testes de usabilidade com usuários
3. ⏳ Coleta de feedback
4. ⏳ Expansão para módulo Empresas
5. ⏳ Expansão para módulo Mídias
6. ⏳ Expansão para módulo Usuários

---

**Desenvolvido por**: FreeFi Admin Team  
**Versão**: 1.0  
**Laravel**: 12.0  
**Bootstrap**: 5.x
