# 📋 FLUXO DE CADASTRO: CRIATIVOS E PUBLICAÇÕES

**Sistema FreeFi Admin - Modelo de Franquias**  
**Data:** 17 de Outubro de 2025

---

## 🎯 **CONCEITOS PRINCIPAIS**

### **1. CRIATIVOS (Tabela: `criativos`)**
São os **arquivos de mídia** (imagens, vídeos, HTML, etc.) que serão exibidos nos equipamentos.

**Características:**
- ✅ Arquivo físico único (path no storage)
- ✅ Pode ser reutilizado em múltiplas publicações
- ✅ Pertence a uma empresa (franquia ou cliente)
- ✅ Passa por processo de aprovação (FASE 3)
- ✅ Pode ser bloqueado permanentemente

**Analogia:**
> Criativo = **Produto na prateleira**
> É o arquivo que está disponível para uso

---

### **2. PUBLICAÇÕES (Tabela: `publicacoes`)**
São os **contratos de exibição** que definem ONDE, QUANDO e PARA QUEM um criativo será exibido.

**Características:**
- ✅ Referência a um criativo existente
- ✅ Define período de exibição (data_inicio → data_fim)
- ✅ Define relacionamento comercial (franquia vende → cliente compra)
- ✅ Tem valores comerciais (valor_contrato, impressões)
- ✅ Tem status (ativa, pausada, expirada, cancelada)

**Analogia:**
> Publicação = **Contrato de aluguel**
> Define as regras de uso do criativo

---

## 🔄 **FLUXO COMPLETO DE CADASTRO**

```
┌─────────────────────────────────────────────────────────────────────┐
│                    1️⃣ CADASTRO DE CRIATIVO                           │
└─────────────────────────────────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  User acessa: /midias/create             │
        │  View: resources/views/midias/create     │
        └─────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  FORMULÁRIO SOLICITA:                    │
        │  • Título                                │
        │  • Tipo (image/video)                    │
        │  • Arquivo de Mídia (upload)             │
        │  • Empresa (cliente/franquia)            │
        │  • Serial do Equipamento (opcional)      │
        │  • Posição (1-6, opcional)               │
        │  • Data/Hora Início (opcional)           │
        │  • Data/Hora Fim (opcional)              │
        └─────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  POST → MidiaController@store            │
        │  Route: /midias (POST)                   │
        └─────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  PROCESSAMENTO:                          │
        │  1. Validação dos campos                 │
        │  2. Upload do arquivo                    │
        │  3. Redimensionamento (se imagem)        │
        │  4. Detecção do formato (jpg/png/mp4)    │
        │  5. Mapeamento tipo → tipo               │
        │  6. Auto-aprovação (status_aprovacao)    │
        └─────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  REGISTRO CRIADO EM 'criativos':         │
        │  • id: 1                                 │
        │  • title: "Banner Promoção"              │
        │  • tipo: "imagem"                        │
        │  • formato: "jpg"                        │
        │  • path: "midias/abc123.jpg"             │
        │  • empresa_legado_id: 5                  │
        │  • status_aprovacao: "aprovado"          │
        └─────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  Redirect → /midias (listagem)           │
        └─────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────┐
│                    2️⃣ CADASTRO DE PUBLICAÇÃO                         │
│                    (NÃO IMPLEMENTADO AINDA)                          │
└─────────────────────────────────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  User acessa: /publicacoes/create        │
        │  (Rota e Controller não existem)         │
        └─────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  FORMULÁRIO DEVE SOLICITAR:              │
        │  • Criativo (select de criativos)        │
        │  • Franquia (quem vende)                 │
        │  • Cliente (quem compra)                 │
        │  • Título da campanha                    │
        │  • Data início                           │
        │  • Data fim                              │
        │  • Valor do contrato                     │
        │  • Impressões contratadas                │
        │  • Status (ativa/pausada)                │
        └─────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  POST → PublicacaoController@store       │
        │  (Controller não existe)                 │
        └─────────────────────────────────────────┘
                              ↓
        ┌─────────────────────────────────────────┐
        │  REGISTRO CRIADO EM 'publicacoes':       │
        │  • id: 1                                 │
        │  • criativo_id: 1 (FK)                   │
        │  • franquia_id: 3 (FK)                   │
        │  • cliente_id: 8 (FK)                    │
        │  • titulo: "Campanha Verão 2025"         │
        │  • data_inicio: "2025-01-01"             │
        │  • data_fim: "2025-03-31"                │
        │  • valor_contrato: 5000.00               │
        │  • impressoes_contratadas: 10000         │
        │  • status: "ativa"                       │
        │  • publicada_por: user_id                │
        └─────────────────────────────────────────┘
```

---

## 📊 **ESTRUTURA DA TABELA `criativos`**

```sql
CREATE TABLE `criativos` (
    `id` BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    
    -- Informações Básicas
    `title` VARCHAR(255) NOT NULL,                    -- Título do criativo
    `tipo` ENUM('imagem','video','html','quiz','nps'), -- Tipo do conteúdo
    `formato` VARCHAR(50),                             -- Extensão (jpg, png, mp4)
    `path` VARCHAR(255) NULLABLE,                      -- Caminho do arquivo
    `conteudo_html` TEXT NULLABLE,                     -- HTML customizado
    
    -- Relacionamentos
    `empresa_legado_id` BIGINT UNSIGNED NULLABLE,      -- Empresa dona (FK)
    `franquia_id` BIGINT UNSIGNED NULLABLE,            -- Franquia (FK)
    
    -- Configurações de Exibição (LEGADAS)
    `equipment_serial` VARCHAR(255),                   -- Serial do equipamento
    `position` INT(1-6),                               -- Posição na tela
    `start_datetime` DATETIME,                         -- Início agendado
    `end_datetime` DATETIME,                           -- Fim agendado
    
    -- Controle e Métricas
    `status` BOOLEAN DEFAULT true,                     -- Ativo/Inativo
    `views_count` INT DEFAULT 0,                       -- Visualizações
    
    -- Sistema de Aprovação (FASE 3)
    `status_aprovacao` ENUM('pendente','aprovado','rejeitado') DEFAULT 'pendente',
    `aprovado_por` BIGINT UNSIGNED NULLABLE,           -- User que aprovou (FK)
    `aprovado_em` TIMESTAMP NULLABLE,                  -- Data de aprovação
    `motivo_rejeicao` TEXT NULLABLE,                   -- Motivo se rejeitado
    `bloqueado_permanente` BOOLEAN DEFAULT false,      -- Bloqueio definitivo
    
    -- Userstamps
    `created_by`, `updated_by`, `deleted_by`,
    `created_at`, `updated_at`, `deleted_at`
);
```

---

## 📊 **ESTRUTURA DA TABELA `publicacoes`**

```sql
CREATE TABLE `publicacoes` (
    `id` BIGINT UNSIGNED PRIMARY KEY AUTO_INCREMENT,
    
    -- Relacionamentos Principais
    `criativo_id` BIGINT UNSIGNED NOT NULL,            -- Criativo usado (FK)
    `franquia_id` BIGINT UNSIGNED NOT NULL,            -- Quem vende (FK)
    `cliente_id` BIGINT UNSIGNED NOT NULL,             -- Quem compra (FK)
    
    -- Informações da Campanha
    `titulo` VARCHAR(255) NOT NULL,                    -- Título da campanha
    `data_inicio` DATE NOT NULL,                       -- Início do contrato
    `data_fim` DATE NOT NULL,                          -- Fim do contrato
    
    -- Valores Comerciais
    `valor_contrato` DECIMAL(10,2) NULLABLE,           -- Valor em R$
    `impressoes_contratadas` INT NULLABLE,             -- Nº de exibições
    
    -- Controle
    `status` ENUM('ativa','pausada','expirada','cancelada') DEFAULT 'ativa',
    `publicada_por` BIGINT UNSIGNED NOT NULL,          -- User que criou (FK)
    `publicada_em` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Userstamps
    `created_by`, `updated_by`, `deleted_by`,
    `created_at`, `updated_at`, `deleted_at`,
    
    -- Constraints
    FOREIGN KEY (`criativo_id`) REFERENCES `criativos`(`id`),
    FOREIGN KEY (`franquia_id`) REFERENCES `empresas`(`id`),
    FOREIGN KEY (`cliente_id`) REFERENCES `empresas`(`id`),
    FOREIGN KEY (`publicada_por`) REFERENCES `users`(`id`)
);
```

---

## 🔗 **RELACIONAMENTOS**

```
┌─────────────┐          ┌──────────────┐          ┌─────────────┐
│  empresas   │          │  criativos   │          │ publicacoes │
│  (clientes) │──────────│  (arquivos)  │──────────│ (contratos) │
│             │ 1     N  │              │ 1     N  │             │
└─────────────┘          └──────────────┘          └─────────────┘
      │                         │                          │
      │                         │                          │
      │ tipo='franquia'         │                          │ franquia_id
      │                         │                          │ cliente_id
      │                         │ empresa_legado_id        │
      │                         │ franquia_id              │
      └─────────────────────────┴──────────────────────────┘

┌──────────────────────────────────────────────────────────────────┐
│  EXEMPLO DE RELACIONAMENTO:                                       │
├──────────────────────────────────────────────────────────────────┤
│  Empresa "FreeFi Matriz" (id=1, tipo=matriz)                     │
│  └── Criativo "Logo FreeFi" (empresa_legado_id=NULL)             │
│      └── Publicação #1: Franquia SP vende para Cliente X         │
│      └── Publicação #2: Franquia RJ vende para Cliente Y         │
│                                                                   │
│  Empresa "Franquia SP" (id=2, tipo=franquia, franquia_pai=1)    │
│  └── Criativo "Banner Promoção SP"                               │
│      └── Publicação #3: Franquia SP vende para Cliente Z         │
└──────────────────────────────────────────────────────────────────┘
```

---

## 💻 **CÓDIGO ATUAL - MidiaController@store**

```php
public function store(Request $request)
{
    // 1. VALIDAÇÃO
    $request->validate([
        'title' => 'required|string|max:255',
        'tipo' => 'nullable|in:imagem,video,html,quiz,nps',  // Novo
        'type' => 'nullable|in:image,video',                 // Legado
        'media_file' => 'nullable|file',
        'empresa_legado_id' => 'required|exists:empresas,id',
        'equipment_serial' => 'nullable|string|max:255',
        'position' => 'nullable|integer|between:1,6',
        'start_datetime' => 'nullable|date',
        'end_datetime' => 'nullable|date|after:start_datetime',
    ]);

    // 2. INICIALIZAÇÃO
    $path = null;
    $tipo = $request->tipo ?? ($request->type === 'image' ? 'imagem' : 'video');
    $formato = 'jpg'; // default
    
    // 3. UPLOAD E PROCESSAMENTO DO ARQUIVO
    if ($request->hasFile('media_file')) {
        $extension = $request->file('media_file')->extension();
        $formato = in_array($extension, ['jpg', 'png', 'gif', 'mp4', 'html']) 
                   ? $extension 
                   : 'jpg';
        
        // Se for IMAGEM: redimensiona
        if ($request->type === 'image' || $tipo === 'imagem') {
            $manager = new ImageManager(new Driver());
            $image = $manager->read($request->file('media_file'));
            $image->resize(640, 1100);  // Dimensões fixas
            $filename = uniqid() . '.' . $extension;
            $path = 'midias/' . $filename;
            Storage::disk('public')->put($path, (string) $image->encode());
        } 
        // Se for VÍDEO: apenas move
        else {
            $path = $request->file('media_file')->store('midias', 'public');
        }
    }

    // 4. CRIAÇÃO DO REGISTRO
    Midia::create([
        'title' => $request->title,
        'tipo' => $tipo,
        'formato' => $formato,
        'path' => $path,
        'empresa_legado_id' => $request->empresa_legado_id,
        'equipment_serial' => $request->equipment_serial,
        'position' => $request->position,
        'start_datetime' => $request->start_datetime,
        'end_datetime' => $request->end_datetime,
        'status_aprovacao' => 'aprovado',  // ⚠️ AUTO-APROVADO (temporário)
    ]);

    // 5. REDIRECT
    return redirect()->route('midias.index');
}
```

---

## 📝 **CAMPOS DO FORMULÁRIO DE CRIATIVO**

### **Campos Obrigatórios:**
| Campo | Tipo | Descrição |
|-------|------|-----------|
| `title` | text | Título do criativo |
| `type` | select | image / video |
| `media_file` | file | Arquivo a fazer upload |
| `empresa_legado_id` | select | Empresa proprietária |

### **Campos Opcionais:**
| Campo | Tipo | Descrição |
|-------|------|-----------|
| `equipment_serial` | text | Serial do equipamento |
| `position` | select | Posição 1-6 |
| `start_datetime` | datetime | Data/hora de início |
| `end_datetime` | datetime | Data/hora de fim |

---

## ⚠️ **OBSERVAÇÕES IMPORTANTES**

### **1. CAMPOS LEGADOS vs NOVOS:**
- ✅ `empresa_id` → `empresa_legado_id` (renomeado)
- ✅ `type` → `tipo` (renomeado, valores em português)
- ✅ Formulário ainda usa `type` (compatibilidade)
- ✅ Controller mapeia: `type='image'` → `tipo='imagem'`

### **2. AUTO-APROVAÇÃO (TEMPORÁRIO):**
```php
'status_aprovacao' => 'aprovado',  // ⚠️ Sempre aprovado
```
**Motivo:** Sistema de aprovação (FASE 3) não implementado ainda.

**FASE 3 terá:**
- Status padrão: `'pendente'`
- Interface de aprovação para SuperAdmin
- Emails de notificação
- Histórico de aprovações

### **3. DIMENSÕES FIXAS DE IMAGEM:**
```php
$image->resize(640, 1100);  // Largura x Altura
```
**Motivo:** Padronização para equipamentos/displays.

### **4. PUBLICAÇÕES NÃO IMPLEMENTADAS:**
❌ Não existe `PublicacaoController`  
❌ Não existe rota `/publicacoes`  
❌ Não existe view de publicações  
❌ Tabela `publicacoes` criada mas não usada

**Para implementar:**
```bash
php artisan make:controller PublicacaoController --resource
```

---

## 🎯 **FLUXO IDEAL (FASE 3)**

```
1. CRIATIVO É CRIADO
   └─ status_aprovacao = 'pendente'
   └─ Notifica SuperAdmin via email

2. SUPERADMIN APROVA/REJEITA
   └─ Se aprovado: status_aprovacao = 'aprovado'
   └─ Se rejeitado: status_aprovacao = 'rejeitado' + motivo
   └─ Notifica criador via email

3. CRIATIVO APROVADO FICA DISPONÍVEL
   └─ Aparece no catálogo de criativos
   └─ Pode ser usado em publicações

4. FRANQUIA CRIA PUBLICAÇÃO
   └─ Seleciona criativo aprovado
   └─ Define cliente, período, valores
   └─ Publicação entra em modo 'ativa'

5. EQUIPAMENTOS BUSCAM PUBLICAÇÕES
   └─ Query: status='ativa' + data atual entre data_inicio e data_fim
   └─ Carrega criativo via criativo_id
   └─ Exibe no display
   └─ Incrementa views_count
```

---

## 📋 **CHECKLIST PARA IMPLEMENTAR PUBLICAÇÕES**

### **Backend:**
- [ ] Criar `PublicacaoController`
- [ ] Completar Model `Publicacao` com:
  - [ ] `$fillable`
  - [ ] Relacionamentos: `criativo()`, `franquia()`, `cliente()`, `publicadoPor()`
  - [ ] Scopes: `ativas()`, `expiradas()`, `porPeriodo()`
  - [ ] Mutators/Accessors se necessário
- [ ] Criar rotas em `routes/web.php`:
  ```php
  Route::resource('publicacoes', PublicacaoController::class);
  ```

### **Frontend:**
- [ ] Criar `resources/views/publicacoes/index.blade.php`
- [ ] Criar `resources/views/publicacoes/create.blade.php`
- [ ] Criar `resources/views/publicacoes/edit.blade.php`
- [ ] Criar `resources/views/publicacoes/show.blade.php`
- [ ] Adicionar menu no sidebar
- [ ] Criar cards de dashboard (publicações ativas, expiradas, etc.)

### **Validações:**
- [ ] Validar datas (data_fim > data_inicio)
- [ ] Validar que criativo está aprovado
- [ ] Validar que franquia/cliente existem
- [ ] Validar valores comerciais
- [ ] Impedir overlap de publicações no mesmo equipamento

### **Business Rules:**
- [ ] Implementar mudança automática de status:
  - `ativa` → `expirada` (quando data_fim < hoje)
- [ ] Job/Command para verificar expiração diariamente
- [ ] Notificações de publicação próxima do fim
- [ ] Dashboard com KPIs (publicações ativas, faturamento, etc.)

---

## 🎨 **EXEMPLO PRÁTICO**

### **Cenário: Franquia SP vende para Cliente XYZ**

**1. Criativo já existe:**
```
ID: 5
Título: "Banner Black Friday"
Tipo: imagem
Path: midias/abc123.jpg
Status: aprovado
```

**2. Criar Publicação:**
```php
Publicacao::create([
    'criativo_id' => 5,                    // Criativo existente
    'franquia_id' => 10,                   // Franquia SP
    'cliente_id' => 25,                    // Cliente XYZ
    'titulo' => 'Campanha Black Friday XYZ',
    'data_inicio' => '2025-11-20',
    'data_fim' => '2025-11-30',
    'valor_contrato' => 3000.00,
    'impressoes_contratadas' => 5000,
    'status' => 'ativa',
    'publicada_por' => auth()->id(),
]);
```

**3. Equipamento do Cliente XYZ busca:**
```php
$publicacoes = Publicacao::where('cliente_id', 25)
    ->where('status', 'ativa')
    ->whereDate('data_inicio', '<=', now())
    ->whereDate('data_fim', '>=', now())
    ->with('criativo')
    ->get();

foreach ($publicacoes as $pub) {
    $criativo = $pub->criativo;  // Carrega o arquivo
    // Exibe: $criativo->path = "midias/abc123.jpg"
}
```

---

## ✅ **RESUMO**

| Conceito | Status | O que é |
|----------|--------|---------|
| **Criativos** | ✅ Implementado | Arquivo de mídia (imagem/vídeo) |
| **Publicações** | ❌ Não implementado | Contrato de exibição |
| **Aprovação** | ⚠️ Auto-aprovado | Sistema completo (FASE 3) |
| **Dashboard** | ❌ Falta | Métricas e KPIs |

**Próximos Passos:**
1. Implementar sistema de aprovação (FASE 3)
2. Criar CRUD de Publicações
3. Criar dashboard com métricas
4. Implementar API para equipamentos

---

**Documentação atualizada em:** 17 de Outubro de 2025  
**Sistema:** FreeFi Admin v1.0
