# 🔌 Guia de Integração: Laravel Admin + Mikrotik Hotspot

**Sistema:** FreeFi Admin (Laravel)  
**Target:** Mikrotik RouterOS (Hotspot)  
**Data:** 19/10/2025

---

## 🎯 Visão Geral da Integração

```
┌─────────────────────────────────────────────────────────────┐
│                    ARQUITETURA COMPLETA                     │
├─────────────────────────────────────────────────────────────┤
│                                                             │
│  ┌──────────────┐         ┌──────────────┐                │
│  │   MIKROTIK   │────────▶│ LARAVEL API  │                │
│  │   HOTSPOT    │  HTTPS  │  (FreeFi)    │                │
│  │              │◀────────│              │                │
│  └──────────────┘         └──────────────┘                │
│         │                        │                         │
│         │ Portal Cativo          │ Admin Web               │
│         ▼                        ▼                         │
│  ┌──────────────┐         ┌──────────────┐                │
│  │   USUÁRIO    │         │  FRANQUIA/   │                │
│  │   (WiFi)     │         │   CLIENTE    │                │
│  └──────────────┘         └──────────────┘                │
│                                                             │
└─────────────────────────────────────────────────────────────┘
```

---

## 🔧 Como Funciona a Integração

### 1️⃣ **Mikrotik Hotspot → Laravel API**

O Mikrotik vai fazer chamadas HTTP para a API Laravel em 3 momentos:

#### A) **Heartbeat** (A cada 5 minutos)
```bash
# Script no Mikrotik RouterOS
/system script add name=freefi_heartbeat source={
  :local serial [/system identity get name];
  :local vpnip [/ip address get [find interface="vpn-interface"] address];
  
  /tool fetch \
    url="https://admin.freefi.com.br/api/hotspot/heartbeat" \
    http-method=post \
    http-header-field="Content-Type: application/json" \
    http-data="{\"serial\":\"$serial\",\"vpn_ip\":\"$vpnip\"}" \
    mode=https \
    keep-result=no
}

# Agendar execução
/system scheduler add name=freefi_heartbeat \
  interval=5m \
  on-event=freefi_heartbeat
```

**O que isso faz:**
- ✅ Registra que o hotspot está online
- ✅ Atualiza campo `last_activity_at`
- ✅ Laravel marca status como "online"

---

#### B) **Buscar Publicações** (A cada 1 hora ou on-demand)
```bash
# Script no Mikrotik RouterOS
/system script add name=freefi_sync_publicacoes source={
  :local serial [/system identity get name];
  
  /tool fetch \
    url="https://admin.freefi.com.br/api/hotspot/$serial/publicacoes" \
    http-method=get \
    mode=https \
    dst-path=publicacoes.json
  
  # Processar JSON e atualizar portal cativo
  # (código específico depende da implementação do portal)
}
```

**Resposta da API:**
```json
{
  "publicacoes": [
    {
      "id": 1,
      "titulo": "Black Friday - 50% OFF",
      "criativo": {
        "id": 1,
        "title": "Banner Black Friday",
        "path": "criativos/banner_blackfriday.jpg",
        "url": "https://admin.freefi.com.br/storage/criativos/banner_blackfriday.jpg"
      },
      "link_destino": "https://www.loja.com.br/blackfriday",
      "habilitar_clique": true,
      "track_impressoes": true,
      "track_cliques": true,
      "walled_gardens": [
        {
          "type": "domain",
          "value": "loja.com.br",
          "ativo": true
        },
        {
          "type": "domain",
          "value": "www.loja.com.br",
          "ativo": true
        }
      ],
      "data_inicio": "2025-11-20",
      "data_fim": "2025-11-27"
    }
  ]
}
```

**O que o Mikrotik faz com isso:**
1. Baixa imagens dos criativos
2. Atualiza portal cativo com novo banner
3. Configura walled gardens (libera domínios)

---

#### C) **Registrar Impressões/Cliques** (Quando usuário interage)

**Quando usuário VÊ o banner:**
```bash
# No portal cativo (HTML/JavaScript)
<script>
// Quando página carrega
fetch('https://admin.freefi.com.br/api/hotspot/impressao', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({
    publicacao_id: 1,
    hotspot_id: 3,
    criativo_id: 1,
    data: '2025-10-20',
    impressoes_unicas: 1
  })
});
</script>
```

**Quando usuário CLICA no banner:**
```html
<a href="#" onclick="registrarClique(1)">
  <img src="banner.jpg">
</a>

<script>
function registrarClique(publicacaoId) {
  fetch('https://admin.freefi.com.br/api/hotspot/clique', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      publicacao_id: publicacaoId,
      hotspot_id: 3,
      criativo_id: 1,
      user_ip: getUserIP(),
      user_mac: getUserMAC()
    })
  }).then(() => {
    // Redireciona para o link
    window.location.href = 'https://www.loja.com.br/blackfriday';
  });
}
</script>
```

---

## 🛡️ Walled Gardens no Mikrotik

### O que são Walled Gardens?

No Mikrotik, walled gardens são domínios/IPs que o usuário pode acessar **ANTES** de fazer login no hotspot.

### Como Aplicar?

```bash
# Quando o Mikrotik recebe walled gardens da API, executa:

# Para cada domain
/ip hotspot walled-garden add \
  dst-host="loja.com.br" \
  action=allow \
  comment="Publicacao ID 1"

/ip hotspot walled-garden add \
  dst-host="*.loja.com.br" \
  action=allow \
  comment="Publicacao ID 1"

# Para cada IP
/ip hotspot walled-garden ip add \
  dst-address=200.150.30.100 \
  action=allow \
  comment="Publicacao ID 1"

# Para cada subnet
/ip hotspot walled-garden ip add \
  dst-address=192.168.1.0/24 \
  action=allow \
  comment="Publicacao ID 1"
```

**Quando publicação expira:**
```bash
# Remover walled gardens antigos
/ip hotspot walled-garden remove [find comment="Publicacao ID 1"]
/ip hotspot walled-garden ip remove [find comment="Publicacao ID 1"]
```

---

## 📱 Portal Cativo - Exemplo de Implementação

### Estrutura do Portal Cativo

```html
<!DOCTYPE html>
<html>
<head>
  <title>WiFi Grátis - FreeFi</title>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <style>
    body {
      font-family: Arial, sans-serif;
      margin: 0;
      padding: 20px;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    }
    
    .container {
      max-width: 600px;
      margin: 0 auto;
      background: white;
      border-radius: 20px;
      padding: 30px;
      box-shadow: 0 10px 40px rgba(0,0,0,0.2);
    }
    
    .banner-container {
      margin-bottom: 30px;
      text-align: center;
    }
    
    .banner {
      max-width: 100%;
      border-radius: 10px;
      cursor: pointer;
      transition: transform 0.3s;
    }
    
    .banner:hover {
      transform: scale(1.05);
    }
    
    .login-form {
      text-align: center;
    }
    
    .btn {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: white;
      border: none;
      padding: 15px 40px;
      border-radius: 10px;
      font-size: 18px;
      cursor: pointer;
      width: 100%;
    }
  </style>
</head>
<body>
  <div class="container">
    <!-- Banner da Publicação -->
    <div class="banner-container" id="bannerContainer">
      <!-- Carregado via JavaScript -->
    </div>
    
    <!-- Formulário de Login -->
    <div class="login-form">
      <h2>WiFi Grátis</h2>
      <p>Clique abaixo para conectar</p>
      
      <form method="post" action="$(link-login-only)">
        <input type="hidden" name="username" value="guest">
        <input type="hidden" name="password" value="">
        <input type="hidden" name="dst" value="$(link-orig)">
        <input type="hidden" name="popup" value="true">
        <button type="submit" class="btn">Conectar Agora</button>
      </form>
    </div>
  </div>

  <script>
    // Configurações (vem do Mikrotik ou arquivo local)
    const HOTSPOT_ID = 3; // ID do hotspot no banco Laravel
    const API_URL = 'https://admin.freefi.com.br/api/hotspot';
    
    // Buscar publicação vigente
    async function carregarPublicacao() {
      try {
        const response = await fetch(`${API_URL}/E2E-TEST-001/publicacoes`);
        const data = await response.json();
        
        if (data.publicacoes && data.publicacoes.length > 0) {
          const pub = data.publicacoes[0]; // Primeira publicação
          mostrarBanner(pub);
          registrarImpressao(pub.id, pub.criativo.id);
        }
      } catch (error) {
        console.error('Erro ao carregar publicação:', error);
      }
    }
    
    // Mostrar banner
    function mostrarBanner(publicacao) {
      const container = document.getElementById('bannerContainer');
      
      const img = document.createElement('img');
      img.src = publicacao.criativo.url;
      img.alt = publicacao.titulo;
      img.className = 'banner';
      
      if (publicacao.habilitar_clique) {
        img.onclick = () => registrarCliqueEAbrir(publicacao);
      }
      
      container.appendChild(img);
    }
    
    // Registrar impressão
    async function registrarImpressao(publicacaoId, criativoId) {
      try {
        await fetch(`${API_URL}/impressao`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            publicacao_id: publicacaoId,
            hotspot_id: HOTSPOT_ID,
            criativo_id: criativoId,
            data: new Date().toISOString().split('T')[0],
            impressoes_unicas: 1
          })
        });
      } catch (error) {
        console.error('Erro ao registrar impressão:', error);
      }
    }
    
    // Registrar clique
    async function registrarCliqueEAbrir(publicacao) {
      try {
        await fetch(`${API_URL}/clique`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            publicacao_id: publicacao.id,
            hotspot_id: HOTSPOT_ID,
            criativo_id: publicacao.criativo.id,
            user_ip: '$(ip)',  // Variável do Mikrotik
            user_mac: '$(mac)' // Variável do Mikrotik
          })
        });
        
        // Abrir link em nova aba
        window.open(publicacao.link_destino, '_blank');
      } catch (error) {
        console.error('Erro ao registrar clique:', error);
      }
    }
    
    // Carregar publicação ao abrir página
    carregarPublicacao();
  </script>
</body>
</html>
```

---

## 🔐 Segurança e HTTPS

### Problema: Mikrotik precisa confiar no certificado SSL

**Solução 1: Certificado Let's Encrypt (Recomendado)**
```bash
# No servidor Laravel
certbot --nginx -d admin.freefi.com.br

# No Mikrotik, importar certificado CA
/tool fetch url="https://letsencrypt.org/certs/isrgrootx1.pem" mode=https
/certificate import file-name=isrgrootx1.pem passphrase=""
```

**Solução 2: Desabilitar verificação SSL (Não recomendado para produção)**
```bash
/tool fetch \
  url="https://admin.freefi.com.br/api/hotspot/heartbeat" \
  http-method=post \
  mode=https \
  check-certificate=no  # ⚠️ Apenas para testes!
```

---

## 📊 Fluxo Completo de Integração

```
┌─────────────────────────────────────────────────────────────┐
│ 1. MIKROTIK LIGA E SINCRONIZA                               │
├─────────────────────────────────────────────────────────────┤
│ [Mikrotik] → POST /api/hotspot/heartbeat                   │
│            ← 200 OK { "message": "Heartbeat registrado" }   │
│                                                             │
│ [Mikrotik] → GET /api/hotspot/HS-001/publicacoes          │
│            ← 200 OK { "publicacoes": [...] }               │
│                                                             │
│ [Mikrotik] Processa JSON:                                  │
│   - Baixa imagens dos criativos                            │
│   - Atualiza portal cativo HTML                            │
│   - Configura walled gardens                               │
└─────────────────────────────────────────────────────────────┘
                             ⬇️
┌─────────────────────────────────────────────────────────────┐
│ 2. USUÁRIO CONECTA NO WIFI                                  │
├─────────────────────────────────────────────────────────────┤
│ [Usuário] Abre navegador                                    │
│ [Mikrotik] Redireciona para portal cativo                   │
│ [Portal] Carrega HTML com banner                            │
│ [Portal JS] → POST /api/hotspot/impressao                  │
│            ← 200 OK { "message": "Impressão registrada" }   │
└─────────────────────────────────────────────────────────────┘
                             ⬇️
┌─────────────────────────────────────────────────────────────┐
│ 3. USUÁRIO CLICA NO BANNER                                  │
├─────────────────────────────────────────────────────────────┤
│ [Portal JS] → POST /api/hotspot/clique                     │
│            ← 200 OK { "id": 1, "message": "Clique..." }    │
│                                                             │
│ [Portal JS] Abre link: www.loja.com.br/blackfriday         │
│ [Mikrotik] Permite acesso (walled garden configurado)      │
│ [Usuário] Navega no site da loja                            │
└─────────────────────────────────────────────────────────────┘
                             ⬇️
┌─────────────────────────────────────────────────────────────┐
│ 4. CLIENTE VÊ ESTATÍSTICAS                                  │
├─────────────────────────────────────────────────────────────┤
│ [Cliente] Acessa: /publicacoes/1/estatisticas              │
│ [Laravel] Mostra:                                           │
│   📊 15.234 impressões                                      │
│   👆 1.234 cliques (8.1% CTR)                              │
│   📈 Gráficos Chart.js                                      │
│   🏆 Top 10 hotspots                                        │
└─────────────────────────────────────────────────────────────┘
```

---

## 🛠️ Implementação Prática no Mikrotik

### Estrutura de Arquivos no Mikrotik

```
/files/
├── hotspot/
│   ├── login.html              ← Portal cativo
│   ├── publicacoes.json        ← Cache de publicações
│   └── criativos/
│       ├── banner1.jpg         ← Imagens baixadas
│       └── banner2.jpg
└── scripts/
    ├── freefi_heartbeat.rsc    ← Script de heartbeat
    ├── freefi_sync.rsc         ← Script de sincronização
    └── freefi_walled.rsc       ← Gerenciar walled gardens
```

### Script Completo de Sincronização

```bash
# /files/scripts/freefi_sync.rsc
:local apiUrl "https://admin.freefi.com.br/api/hotspot"
:local serial [/system identity get name]
:local hotspoId "3"  # ID do hotspot no banco Laravel

# 1. Heartbeat
:do {
  :local vpnip [/ip address get [find interface="vpn"] address]
  /tool fetch \
    url="$apiUrl/heartbeat" \
    http-method=post \
    http-header-field="Content-Type: application/json" \
    http-data="{\"serial\":\"$serial\",\"vpn_ip\":\"$vpnip\"}" \
    mode=https \
    keep-result=no
  :log info "FreeFi: Heartbeat enviado"
} on-error={
  :log error "FreeFi: Erro no heartbeat"
}

# 2. Buscar publicações
:do {
  /tool fetch \
    url="$apiUrl/$serial/publicacoes" \
    http-method=get \
    mode=https \
    dst-path=hotspot/publicacoes.json
  
  :log info "FreeFi: Publicações atualizadas"
  
  # 3. Processar JSON e baixar criativos
  # (Implementação específica depende do parser JSON disponível)
  
} on-error={
  :log error "FreeFi: Erro ao buscar publicações"
}

# 4. Atualizar walled gardens
:do {
  # Remover walled gardens antigos do FreeFi
  /ip hotspot walled-garden remove [find comment~"FreeFi"]
  /ip hotspot walled-garden ip remove [find comment~"FreeFi"]
  
  # Adicionar novos (baseado no JSON)
  # Exemplo:
  /ip hotspot walled-garden add \
    dst-host="loja.com.br" \
    action=allow \
    comment="FreeFi-Pub-1"
    
  :log info "FreeFi: Walled gardens atualizados"
  
} on-error={
  :log error "FreeFi: Erro ao atualizar walled gardens"
}
```

### Agendar Execução

```bash
# Heartbeat a cada 5 minutos
/system scheduler add \
  name=freefi-heartbeat \
  interval=5m \
  on-event="/import file=scripts/freefi_sync.rsc"

# Sincronização completa a cada 1 hora
/system scheduler add \
  name=freefi-sync \
  interval=1h \
  on-event="/import file=scripts/freefi_sync.rsc"
```

---

## 📝 Checklist de Implementação

### No Servidor Laravel (Já Feito ✅)

- [x] API de heartbeat
- [x] API de publicações
- [x] API de walled gardens
- [x] API de impressões
- [x] API de cliques
- [x] Dashboards de estatísticas

### No Mikrotik (Pendente)

- [ ] Criar scripts de sincronização
- [ ] Configurar scheduler
- [ ] Implementar portal cativo HTML
- [ ] Adicionar JavaScript de tracking
- [ ] Testar chamadas HTTP/HTTPS
- [ ] Configurar certificados SSL
- [ ] Implementar parser JSON
- [ ] Automatizar download de criativos
- [ ] Gerenciar walled gardens dinamicamente

### Testes de Integração

- [ ] Heartbeat funciona
- [ ] Publicações são baixadas
- [ ] Walled gardens são aplicados
- [ ] Impressões são registradas
- [ ] Cliques são registrados
- [ ] Portal cativo carrega banner
- [ ] Redirecionamento funciona
- [ ] Estatísticas atualizam em tempo real

---

## 🚨 Problemas Comuns e Soluções

### 1. Mikrotik não consegue chamar API (HTTPS)

**Problema:** Erro de certificado SSL

**Solução:**
```bash
# Importar certificado CA no Mikrotik
/tool fetch url="https://curl.se/ca/cacert.pem" mode=https
/certificate import file-name=cacert.pem passphrase=""

# Ou desabilitar verificação (apenas teste)
check-certificate=no
```

---

### 2. Portal cativo não registra impressões

**Problema:** CORS bloqueando requisições

**Solução no Laravel:**
```php
// config/cors.php
'paths' => ['api/*'],
'allowed_origins' => ['*'],  // Ou IP do Mikrotik
'allowed_methods' => ['*'],
'allowed_headers' => ['*'],
```

---

### 3. Walled garden não funciona

**Problema:** Domínio não foi adicionado corretamente

**Solução:**
```bash
# Adicionar também wildcard
/ip hotspot walled-garden add dst-host="*.loja.com.br" action=allow
/ip hotspot walled-garden add dst-host="loja.com.br" action=allow

# Verificar lista
/ip hotspot walled-garden print
```

---

### 4. JSON parsing no RouterOS

**Problema:** RouterOS não tem parser JSON nativo

**Solução 1: Usar container (RouterOS 7+)**
```bash
# Criar container com Python para processar JSON
/container add \
  remote-image=python:alpine \
  interface=veth1
```

**Solução 2: API retorna script RouterOS**
```bash
# Laravel retorna direto em formato .rsc
GET /api/hotspot/HS-001/publicacoes?format=rsc

# Resposta:
/ip hotspot walled-garden add dst-host="loja.com.br" comment="Pub-1"
/ip hotspot walled-garden add dst-host="*.loja.com.br" comment="Pub-1"
```

---

## 🎯 Próximos Passos

1. **Amanhã:** Testar interface Laravel (conforme planejamento)
2. **Depois:** Implementar scripts no Mikrotik
3. **Integração:** Testar comunicação Mikrotik ↔ Laravel
4. **Portal:** Customizar HTML do portal cativo
5. **Produção:** Deploy e monitoramento

---

## 📚 Recursos Úteis

- **Mikrotik API Docs:** https://help.mikrotik.com/docs/
- **RouterOS Scripting:** https://wiki.mikrotik.com/wiki/Manual:Scripting
- **Hotspot Walled Garden:** https://wiki.mikrotik.com/wiki/Manual:IP/Hotspot
- **Tool Fetch:** https://wiki.mikrotik.com/wiki/Manual:Tools/Fetch

---

**Última atualização:** 19/10/2025  
**Versão Laravel:** 12.0  
**Versão RouterOS:** 7.x+
