# 🔍 PROBLEMA: UPLOAD DE CRIATIVOS NÃO SALVA ARQUIVO FÍSICO

## ❌ PROBLEMA IDENTIFICADO

### **Sintomas:**
1. Upload de criativo funciona sem erro
2. Registro é criado no banco de dados
3. Campo `path` fica NULL/vazio no banco
4. Arquivo físico NÃO é salvo no storage
5. Pasta `/storage/app/public/midias/` está vazia

### **Análise do Banco de Dados:**

```sql
SELECT id, title, path, formato, tipo, created_at 
FROM criativos 
ORDER BY id DESC 
LIMIT 3;
```

**Resultado:**
```
ID | Título                           | Path | Formato | Tipo   | Criado
---+----------------------------------+------+---------+--------+--------------------
1  | Banner Black Friday 2025         | NULL | jpg     | imagem | 2025-10-17 17:43:01
2  | Vídeo Institucional FreeFi       | NULL | mp4     | video  | 2025-10-17 17:43:01
3  | Promoção de Verão 2025           | NULL | png     | imagem | 2025-10-17 17:43:01
```

❌ **Campo `path` está NULL para TODOS os registros!**

---

## 🐛 CAUSA RAIZ

### **1. Link Simbólico INCORRETO**

```bash
$ ls -lha /var/www/html/laravel/admin/freefi_admin/public/ | grep storage

lrwxrwxrwx  1 cloud cloud 32 out 10 storage -> /var/www/html/storage/app/public
```

❌ **PROBLEMA:** O link aponta para `/var/www/html/storage/app/public`  
✅ **DEVERIA:** Apontar para `/var/www/html/laravel/admin/freefi_admin/storage/app/public`

### **2. Seeders criaram registros SEM arquivo**

Os registros foram criados via seeder/factory sem fazer upload real de arquivo, então o campo `path` ficou NULL.

### **3. Controller salva no caminho errado**

```php
// MidiaController.php linha 62-64
if ($request->type === 'image' || $tipo === 'imagem') {
    $path = 'midias/' . $filename;
    Storage::disk('public')->put($path, (string) $image->encode());
}
```

O código está **CORRETO**, mas como o link simbólico está errado, o Laravel não consegue salvar no local correto.

---

## ✅ SOLUÇÃO

### **Passo 1: Remover link simbólico incorreto**

```bash
rm /var/www/html/laravel/admin/freefi_admin/public/storage
```

### **Passo 2: Recriar link simbólico CORRETO**

```bash
cd /var/www/html/laravel/admin/freefi_admin
php artisan storage:link
```

**Saída esperada:**
```
The [public/storage] link has been connected to [storage/app/public].
```

### **Passo 3: Verificar link criado**

```bash
ls -lha /var/www/html/laravel/admin/freefi_admin/public/ | grep storage
```

**Saída esperada:**
```
lrwxrwxrwx 1 cloud cloud 47 out 18 storage -> /var/www/html/laravel/admin/freefi_admin/storage/app/public
```

✅ Agora o link aponta para o caminho **CORRETO**!

### **Passo 4: Testar upload**

```bash
# 1. Acessar http://localhost:8082/midias/create
# 2. Fazer upload de uma imagem
# 3. Verificar no banco se path foi preenchido
# 4. Verificar se arquivo físico foi salvo em storage/app/public/midias/
```

---

## 📊 ESTRUTURA CORRETA DO STORAGE

### **Estrutura esperada:**

```
/var/www/html/laravel/admin/freefi_admin/
├── public/
│   └── storage/  → (link simbólico para ../storage/app/public)
│
└── storage/
    └── app/
        └── public/
            ├── logos/
            │   └── logo_empresa_123.jpg
            └── midias/
                ├── 67890abc.jpg  ← Arquivos reais aqui
                ├── 12345xyz.png
                └── video_teste.mp4
```

### **URLs públicas:**

```
http://localhost:8082/storage/midias/67890abc.jpg
http://localhost:8082/storage/logos/logo_empresa_123.jpg
```

### **Caminhos no banco:**

```sql
-- Campo 'path' na tabela 'criativos'
midias/67890abc.jpg
midias/12345xyz.png
midias/video_teste.mp4

-- Campo 'logomarca' na tabela 'empresas'
logos/logo_empresa_123.jpg
```

---

## 🔧 COMANDOS ÚTEIS

### **Verificar link simbólico atual:**
```bash
ls -lha public/storage
```

### **Recriar link (com força):**
```bash
rm public/storage
php artisan storage:link
```

### **Listar arquivos no storage:**
```bash
ls -lha storage/app/public/midias/
ls -lha storage/app/public/logos/
```

### **Verificar permissões:**
```bash
chmod -R 775 storage/
chmod -R 775 public/
```

### **Verificar se Laravel consegue escrever:**
```bash
touch storage/app/public/midias/test.txt
ls -lha storage/app/public/midias/test.txt
rm storage/app/public/midias/test.txt
```

---

## 🧪 TESTE COMPLETO

### **1. Verificar registros existentes (todos com path NULL):**

```bash
docker exec freefi-admin-app php artisan tinker --execute="DB::table('criativos')->get(['id', 'title', 'path'])->each(fn(\$c) => echo \$c->id . ' | ' . \$c->title . ' | ' . (\$c->path ?? 'NULL') . PHP_EOL);"
```

### **2. Criar novo criativo via interface:**

1. Acessar: http://localhost:8082/midias/create
2. Preencher formulário:
   - Título: "Teste Upload Real"
   - Tipo: Imagem
   - Arquivo: Escolher uma imagem JPG/PNG
   - Empresa: Selecionar qualquer
3. Salvar

### **3. Verificar se salvou corretamente:**

```bash
# Banco de dados
docker exec freefi-admin-app php artisan tinker --execute="echo DB::table('criativos')->latest()->first()->path;"

# Arquivo físico
ls -lha storage/app/public/midias/
```

**Resultado esperado:**
```
# Banco
midias/67890abc123.jpg

# Arquivo físico
-rw-r--r-- 1 www-data www-data 45K out 18 14:30 67890abc123.jpg
```

---

## 📝 CHECKLIST DE CORREÇÃO

- [ ] Remover link simbólico incorreto: `rm public/storage`
- [ ] Recriar link correto: `php artisan storage:link`
- [ ] Verificar link criado: `ls -lha public/storage`
- [ ] Verificar permissões: `chmod -R 775 storage/`
- [ ] Testar upload de imagem via interface
- [ ] Verificar campo `path` no banco após upload
- [ ] Verificar arquivo físico em `storage/app/public/midias/`
- [ ] Testar URL pública: `http://localhost:8082/storage/midias/xxx.jpg`

---

## 🎯 STATUS

**Problema:** ❌ Link simbólico incorreto  
**Impacto:** ❌ Uploads não salvam arquivos físicos  
**Solução:** ✅ Recriar link com `php artisan storage:link`  
**Prioridade:** 🔴 **CRÍTICO** (funcionalidade de upload não funciona)

---

**Criado em:** 18/10/2025  
**Sistema:** FreeFi Admin  
**Módulo:** Criativos (Midias)
