# 🧪 Guia de Testes - Sistema de Emails

Este documento contém instruções detalhadas para testar o sistema de emails automáticos do FreeFi Admin.

---

## 📋 Pré-requisitos

1. ✅ Sistema instalado e funcionando
2. ✅ Banco de dados configurado
3. ✅ Tabela `jobs` criada (migrations executadas)
4. ✅ Pelo menos 1 usuário admin no sistema

---

## 🔧 Configuração para Teste

### Opção 1: Modo Log (Desenvolvimento)

**Vantagens:** Não precisa de SMTP, ideal para desenvolvimento  
**Desvantagens:** Emails não são enviados de verdade

```env
MAIL_MAILER=log
QUEUE_CONNECTION=sync
```

**Onde ver os emails:**
```bash
tail -f storage/logs/laravel.log
```

### Opção 2: Mailtrap (Recomendado para Testes)

**Vantagens:** Vê emails renderizados, testa deliverability  
**Desvantagens:** Precisa criar conta grátis em mailtrap.io

1. Criar conta em https://mailtrap.io (grátis)
2. Copiar credenciais SMTP
3. Configurar `.env`:

```env
MAIL_MAILER=smtp
MAIL_HOST=sandbox.smtp.mailtrap.io
MAIL_PORT=2525
MAIL_USERNAME=seu_username_aqui
MAIL_PASSWORD=sua_senha_aqui
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS="noreply@freefi-admin.com"
MAIL_FROM_NAME="FreeFi Admin"

QUEUE_CONNECTION=database
```

4. Executar worker:
```bash
php artisan queue:work
```

### Opção 3: Gmail (Produção)

**⚠️ Cuidado:** Use apenas em produção, limite de 500 emails/dia

```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=seu_email@gmail.com
MAIL_PASSWORD=senha_de_aplicativo
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS="noreply@freefi-admin.com"
MAIL_FROM_NAME="FreeFi Admin"

QUEUE_CONNECTION=database
```

**Importante:** Use "Senha de Aplicativo", não a senha normal do Gmail.  
Criar em: https://myaccount.google.com/apppasswords

---

## 🧪 Cenários de Teste

### Teste 1: Email de Novo Cadastro Pendente

**Objetivo:** Verificar que administradores recebem notificação quando novo usuário se cadastra.

**Passos:**

1. **Preparação:**
   ```bash
   # Garantir que há pelo menos 1 admin
   php artisan tinker
   >>> use App\Models\User;
   >>> $admin = User::where('is_superadmin', true)->first();
   >>> $admin->email = 'seu_email_de_teste@example.com'; // Trocar
   >>> $admin->save();
   >>> exit
   ```

2. **Iniciar worker (se usando queue=database):**
   ```bash
   php artisan queue:work
   ```
   Deixar rodando em um terminal separado.

3. **Fazer novo cadastro:**
   - Acessar http://localhost/register
   - Inserir código de convite válido (ex: `ZXHB099306`)
   - Preencher formulário
   - Submeter

4. **Verificar:**
   - Se MAIL_MAILER=log: `tail -f storage/logs/laravel.log`
   - Se Mailtrap: Verificar inbox em mailtrap.io
   - Se Gmail: Verificar caixa de entrada

**Resultado Esperado:**
- ✅ Admin recebe email com título: "🔔 Novo Cadastro Pendente de Aprovação - FreeFi Admin"
- ✅ Email contém nome do novo usuário
- ✅ Email contém empresa, telefone, data/hora
- ✅ Badge "⚠️ REQUER AÇÃO IMEDIATA" visível
- ✅ Botão "Gerenciar Cadastro" funcional

---

### Teste 2: Email de Cadastro Aprovado

**Objetivo:** Verificar que usuário recebe confirmação quando cadastro é aprovado.

**Passos:**

1. **Garantir que worker está rodando:**
   ```bash
   php artisan queue:work
   ```

2. **Login como admin:**
   - Acessar http://localhost/login
   - Email: admin@freefi.com.br
   - Senha: sua_senha_admin

3. **Aprovar cadastro:**
   - Ir para http://localhost/users
   - Localizar usuário com status "Pendente" (badge amarelo)
   - Clicar no dropdown de ações (3 pontinhos)
   - Selecionar "✅ Aprovar"

4. **Verificar:**
   - Mensagem de sucesso aparece no topo da página
   - Email enviado ao usuário aprovado

**Resultado Esperado:**
- ✅ Usuário recebe email com título: "✅ Cadastro Aprovado - FreeFi Admin"
- ✅ Email tem header verde com gradiente
- ✅ Info box com dados do usuário
- ✅ Botão "Acessar Sistema" funcional
- ✅ Dica de trocar senha presente

**Verificar no sistema:**
```bash
php artisan tinker
>>> use App\Models\User;
>>> $user = User::where('status_cadastro', 'aprovado')->latest()->first();
>>> $user->status_cadastro; // Deve retornar "aprovado"
```

---

### Teste 3: Email de Cadastro Rejeitado

**Objetivo:** Verificar que usuário recebe notificação quando cadastro é rejeitado.

**Passos:**

1. **Criar novo usuário pendente** (se necessário):
   ```bash
   php artisan tinker
   >>> use App\Models\{User, Empresa};
   >>> $empresa = Empresa::where('tipo_empresa', 'franquia')->first();
   >>> $user = User::create([
   >>>     'name' => 'Teste Rejeição',
   >>>     'email' => 'teste_rejeicao@example.com',
   >>>     'password' => bcrypt('senha123'),
   >>>     'empresa_id' => $empresa->id,
   >>>     'status_cadastro' => 'pendente',
   >>> ]);
   >>> exit
   ```

2. **Login como admin e rejeitar:**
   - Ir para http://localhost/users
   - Localizar "Teste Rejeição" (pendente)
   - Clicar no dropdown de ações
   - Selecionar "❌ Rejeitar"

3. **Verificar email:**

**Resultado Esperado:**
- ✅ Usuário recebe email com título: "❌ Cadastro Rejeitado - FreeFi Admin"
- ✅ Email tem header vermelho com gradiente
- ✅ Info box com dados do usuário
- ✅ Alert box com motivo (se fornecido)
- ✅ Lista de próximos passos
- ✅ Botão "Tentar Novo Cadastro"

**Verificar no sistema:**
```bash
php artisan tinker
>>> use App\Models\User;
>>> $user = User::where('email', 'teste_rejeicao@example.com')->first();
>>> $user->status_cadastro; // Deve retornar "rejeitado"
```

---

## 🔍 Verificações Adicionais

### Verificar Jobs na Fila

```bash
php artisan tinker
>>> DB::table('jobs')->count();  // Quantidade de jobs pendentes
>>> DB::table('jobs')->get();    // Listar jobs
>>> DB::table('failed_jobs')->count(); // Jobs que falharam
```

### Verificar Logs de Email

```bash
# Ver logs em tempo real
tail -f storage/logs/laravel.log

# Buscar emails específicos
grep "CadastroAprovado" storage/logs/laravel.log
grep "CadastroRejeitado" storage/logs/laravel.log
grep "NovoCadastroPendente" storage/logs/laravel.log
```

### Testar Envio Manual (Tinker)

```bash
php artisan tinker
```

```php
use App\Models\User;
use App\Mail\{CadastroAprovado, CadastroRejeitado, NovoCadastroPendente};
use Illuminate\Support\Facades\Mail;

// Buscar um usuário
$user = User::first();

// Testar CadastroAprovado
Mail::to('seu_email@example.com')->send(new CadastroAprovado($user));

// Testar CadastroRejeitado (com motivo)
Mail::to('seu_email@example.com')->send(new CadastroRejeitado($user, 'Dados incompletos'));

// Testar NovoCadastroPendente
Mail::to('seu_email@example.com')->send(new NovoCadastroPendente($user));

// Enviar com queue (assíncrono)
Mail::to('seu_email@example.com')->queue(new CadastroAprovado($user));
```

---

## 🐛 Troubleshooting

### Problema: Emails não estão sendo enviados

**Solução:**

1. Verificar configuração do .env:
   ```bash
   php artisan config:clear
   php artisan config:cache
   ```

2. Verificar se o worker está rodando:
   ```bash
   # Se QUEUE_CONNECTION=database
   php artisan queue:work
   ```

3. Verificar logs:
   ```bash
   tail -f storage/logs/laravel.log
   ```

4. Testar configuração de email:
   ```bash
   php artisan tinker
   >>> use Illuminate\Support\Facades\Mail;
   >>> Mail::raw('Teste', function($msg) {
   >>>     $msg->to('teste@example.com')->subject('Teste');
   >>> });
   ```

### Problema: Jobs ficam presos na fila

**Solução:**

1. Verificar jobs falhados:
   ```bash
   php artisan queue:failed
   ```

2. Reprocessar job específico:
   ```bash
   php artisan queue:retry {job_id}
   ```

3. Reprocessar todos:
   ```bash
   php artisan queue:retry all
   ```

4. Limpar fila (cuidado!):
   ```bash
   php artisan queue:flush
   ```

### Problema: Middleware bloqueia mesmo usuários aprovados

**Solução:**

```bash
php artisan tinker
>>> use App\Models\User;
>>> $user = User::where('email', 'SEU_EMAIL')->first();
>>> $user->status_cadastro; // Verificar status
>>> $user->update(['status_cadastro' => 'aprovado']); // Forçar aprovação
```

### Problema: Variáveis não aparecem no email

**Verificação:**

1. Verificar que o Mailable está passando variáveis:
   ```php
   // app/Mail/CadastroAprovado.php
   public function content(): Content {
       return new Content(
           view: 'emails.cadastro-aprovado',
           with: [
               'userName' => $this->user->name, // ← Verificar
               // ...
           ]
       );
   }
   ```

2. Verificar que a view está usando as variáveis:
   ```blade
   {{-- resources/views/emails/cadastro-aprovado.blade.php --}}
   <h2>Olá, {{ $userName }}!</h2> {{-- ← Verificar --}}
   ```

---

## 📊 Checklist de Testes Completos

### Antes de Ir para Produção

- [ ] Testar envio de "Novo Cadastro Pendente"
- [ ] Testar envio de "Cadastro Aprovado"
- [ ] Testar envio de "Cadastro Rejeitado"
- [ ] Verificar que emails têm design correto (HTML renderizado)
- [ ] Verificar que botões funcionam (URLs corretas)
- [ ] Testar em diferentes clientes de email (Gmail, Outlook, etc.)
- [ ] Verificar que emails não vão para spam
- [ ] Configurar SMTP de produção (Gmail, SendGrid, etc.)
- [ ] Testar queue:work em produção
- [ ] Configurar supervisor para manter queue:work rodando
- [ ] Verificar logs de erro

### Configurar Supervisor (Produção)

```bash
# /etc/supervisor/conf.d/freefi-worker.conf
[program:freefi-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /var/www/html/laravel/admin/freefi_admin/artisan queue:work --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/var/www/html/laravel/admin/freefi_admin/storage/logs/worker.log
stopwaitsecs=3600
```

```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start freefi-worker:*
```

---

## ✅ Conclusão

Após executar todos os testes acima, você deve ter verificado que:

1. ✅ Novos cadastros notificam administradores
2. ✅ Aprovações notificam usuários com email verde
3. ✅ Rejeições notificam usuários com email vermelho
4. ✅ Emails são enfileirados e enviados corretamente
5. ✅ Design dos emails está correto e responsivo
6. ✅ Todas as variáveis dinâmicas funcionam
7. ✅ Sistema está pronto para produção

---

**Última atualização:** Janeiro 2025  
**Autor:** GitHub Copilot  
**Status:** ✅ Pronto para Testes
